import argparse
import os
from importlib import import_module

from genie.chat import generate_next_turn
from genie.chat import utils as chat_utils

domain_choices = os.listdir("src/genie/domains")
domain_choices = [x for x in domain_choices if not x.startswith("__")]


def create_parser():
    parser = argparse.ArgumentParser(description="Tipbot")
    parser.add_argument(
        "--llm_rewrite",
        action="store_true",
        help="Use LLM to rewrite the user target into statements with variables",
    )
    parser.add_argument(
        "--domain",
        type=str,
        choices=domain_choices,
        default="tip",
    )
    parser.add_argument(
        "--spec_type",
        type=str,
        choices=["python", "spreadsheet"],
        default="python",
    )
    parser.add_argument(
        "--debug",
        action="store_true",
        help="Will prompt the user to save the state of the chatbot",
    )

    return parser


def import_domain(domain, spec_type):
    if domain in domain_choices:
        if spec_type == "python":
            bot = import_module(f"genie.domains.{domain}.worksheets")
        elif spec_type == "spreadsheet":
            spreadsheet = import_module(f"genie.domains.{domain}").spreadsheet
            bot = spreadsheet.gsheet_to_genie(
                bot_name=spreadsheet.botname,
                description=spreadsheet.description,
                prompt_dir=spreadsheet.prompt_dir,
                starting_prompt=spreadsheet.starting_prompt,
                args={},
                api=spreadsheet.api,
                gsheet_id=spreadsheet.gsheet_id_default,
                suql_runner=spreadsheet.suql_runner,
                suql_prompt_selector=spreadsheet.suql_prompt_selector,
            )
        else:
            raise ValueError(f"Spec type {spec_type} not found")

        return bot


def main(**args):
    bot = import_domain(args["domain"], args["spec_type"])
    quit_commands = ["exit", "exit()"]
    while True:
        if len(bot.dlg_history) == 0:
            chat_utils.print_chatbot(bot.starting_prompt)
        user_utterance = None
        if user_utterance is None:
            user_utterance = chat_utils.input_user()
        if user_utterance == quit_commands:
            break

        generate_next_turn(user_utterance, bot)
        chat_utils.print_complete_history(bot.dlg_history)


if __name__ == "__main__":
    parser = create_parser()
    args = parser.parse_args()
    main(**args.__dict__)
