import datetime
import json
import os

import chainlit as cl

from genie.annotation_utils import get_agent_action_schemas, get_context_schema
from genie.chat.agent_policy import run_agent_policy
from genie.chat.dialogue import CurrentDialogueTurn
from genie.chat.response_generator import generate_response
from genie.chat.semantic_parser import semantic_parsing
from genie.environment import GenieContext, GenieRuntime

current_dir = os.path.dirname(os.path.realpath(__file__))

__all__ = [
    "AgentPolicy",
    "CurrentDialogueTurn",
    "ResponseGenerator",
    "SemanticParser",
]


async def generate_next_turn_async(user_utterance: str, bot):
    current_dlg_turn = CurrentDialogueTurn()
    current_dlg_turn.user_utterance = user_utterance

    async with cl.Step(
        name="Processing", type="semantic_parser", language="python"
    ) as step:
        current_dlg_turn.context = GenieContext()
        current_dlg_turn.global_context = GenieContext()
        await cl.make_async(semantic_parsing)(current_dlg_turn, bot.dlg_history, bot)
        step.output = current_dlg_turn.user_target_sp

    async with cl.Step(
        name="Processing", type="agent_policy", language="python", show_input=True
    ) as step:
        await cl.make_async(run_agent_policy)(current_dlg_turn, bot)
        step.input = current_dlg_turn.user_target
        step.output = get_agent_action_schemas(
            current_dlg_turn.system_action, bot.context
        )

    async with cl.Step(
        name="Processing", type="response_generator", language="python"
    ) as step:

        await cl.make_async(generate_response)(current_dlg_turn, bot.dlg_history, bot)
        step.output = get_context_schema(bot.context)
        bot.dlg_history.append(current_dlg_turn)


def generate_next_turn(user_utterance: str, bot):
    current_dlg_turn = CurrentDialogueTurn()
    current_dlg_turn.user_utterance = user_utterance

    current_dlg_turn.context = GenieContext()
    current_dlg_turn.global_context = GenieContext()
    semantic_parsing(current_dlg_turn, bot.dlg_history, bot)

    if current_dlg_turn.user_target is not None:
        run_agent_policy(current_dlg_turn, bot)

    generate_response(current_dlg_turn, bot.dlg_history, bot)
    bot.dlg_history.append(current_dlg_turn)
