import json
import torch
import numpy as np

def numpy_default(obj):
    if isinstance(obj, (np.integer, np.int_, np.intc, np.intp, np.int8, np.int16, np.int32, np.int64)):
        return int(obj)
    elif isinstance(obj, (np.floating, np.float_, np.float16, np.float32, np.float64)):
        return float(obj)
    elif isinstance(obj, (np.ndarray,)):
        return obj.tolist()
    elif isinstance(obj, (np.bool_,)):
        return bool(obj)
    elif isinstance(obj, (np.void,)):
        return None
    elif isinstance(obj, np.dtype):
        return str(obj)
    elif torch is not None and isinstance(obj, torch.Tensor):
        return obj.detach().cpu().numpy().tolist()
    elif hasattr(obj, 'dtype'):
        try:
            return obj.item()
        except (ValueError, AttributeError):
            return str(obj)
    elif isinstance(obj, dict):
        return {str(k): v for k, v in obj.items()}
    try:
        return str(obj)
    except:
        return "UNSERIALIZABLE_OBJECT"

def convert_to_serializable(obj):
    if isinstance(obj, (dict,)):
        return {k: convert_to_serializable(v) for k, v in obj.items()}
    elif isinstance(obj, (list, tuple)):
        return [convert_to_serializable(item) for item in obj]
    else:
        return numpy_default(obj)

class NumpyEncoder(json.JSONEncoder):
    def default(self, obj):
        try:
            converted = convert_to_serializable(obj)
            if isinstance(converted, type(obj)) and not isinstance(obj, (str, int, float, bool, type(None))):
                return super().default(obj)
            return converted
        except Exception as e:
            try:
                return str(obj)
            except Exception:
                return f"UNSERIALIZABLE_OBJECT: {type(obj).__name__}"

def make_array_horizontal(match):
    key_part = match.group(1)
    array_content_str = match.group(2)
    closing_bracket = match.group(3)
    elements = [elem.strip() for elem in array_content_str.split(',') if elem.strip()]
    compact_content = ", ".join(elements)
    return f"{key_part}{compact_content}{closing_bracket}"
