import os
import json
import numpy as np
import random
from typing import List, Tuple
from ..utils.storage import GenerationStorage
from ..utils.evaluator import AlgorithmEvaluator
from .llm_integration import LLMClient
from .individual import AlgorithmIndividual
from ..config.settings import DEFAULT_EVOLUTION_PARAMS
from .evolution_algorithms.tournament import TournamentEvolution
from .evolution_algorithms.differential import DifferentialEvolution
from .operators import InitializeOperator

class EvolutionEngine:
    def __init__(self, problem_path: str):
        self.problem_path = problem_path
        self.storage = GenerationStorage(problem_path)
        self.evaluator = AlgorithmEvaluator(problem_path)
        self.llm_client = LLMClient.from_config(problem_path)
        self.initialize_operator = InitializeOperator(self.llm_client)
        
        config = self._load_problem_config()
        self.evolution_params = {
            **DEFAULT_EVOLUTION_PARAMS, 
            **config.get("evolution_params", {})  
        }
        
        algorithm_type = self.evolution_params.get("algorithm", "TU")
        if algorithm_type == "DE":
            self.evolution_algorithm = DifferentialEvolution(self.evolution_params, self.llm_client)
        elif algorithm_type == "TU":
            self.evolution_algorithm = TournamentEvolution(self.evolution_params, self.llm_client)
        else:
            supported_algorithms = ["DE", "TU"]
            raise ValueError(f"{algorithm_type}, {supported_algorithms}")
        

    def initialize_population(self, size: int) -> List[AlgorithmIndividual]:
        problem_config = self._load_problem_config()
        ideas = self.initialize_operator.ideas_generator.generate_ideas(
            problem_config["description"],
            size
        )

        individuals = []
        for i, idea in enumerate(ideas):
            code = self.initialize_operator.generate_initial_code(
                problem_config["description"],
                problem_config["function_name"],
                problem_config["input_format"],
                problem_config["output_format"],
                idea
            )
            if code:
                individuals.append(AlgorithmIndividual(
                    code,
                    generation=0,
                    metadata={"idea": idea}  
                ))

        return individuals
    
    def run_evolution(self, generations: int = None, population_size: int = None):
        generations = generations or self.evolution_params["generations"]
        population_size = population_size or self.evolution_params["population_size"]
        population = self.initialize_population(population_size)

        for ind in population:
            ind.fitness = self.evaluator.evaluate(ind.code)

        best_ever = min(population, key=lambda x: x.fitness or float('inf'))
        best_fitness = best_ever.fitness
        
        for gen in range(generations):

            new_population = []
            while len(new_population) < population_size:
                num_parents = max(2, int(np.ceil(len(population) * 0.6)))
                parents = self.evolution_algorithm.select(population, num_parents)

                offspring = self.evolution_algorithm.crossover(parents)

                for child in offspring:
                    mutated_child = self.evolution_algorithm.mutate(child)
                    if mutated_child.code:  
                        mutated_child.fitness = self.evaluator.evaluate(mutated_child.code)
                        new_population.append(mutated_child)

            population = self.evolution_algorithm.survive(population, new_population, population_size)

            current_best = min(population, key=lambda x: x.fitness or float('inf'))

            if current_best.fitness < best_fitness:
                best_ever = current_best
                best_fitness = current_best.fitness

            generation_data = {
                "generation": gen,
                "population": [{
                    "code": ind.code,
                    "fitness": ind.fitness,
                    "generation": ind.generation
                } for ind in population],
                "best_fitness": current_best.fitness,
                "best_code": current_best.code,
                "best_ever_fitness": best_fitness,
                "best_ever_code": best_ever.code
            }
            self.storage.save_generation(gen, generation_data)

        return population, best_ever

    def _tournament_select(self, population: List[AlgorithmIndividual], num_winners: int) -> List[AlgorithmIndividual]:

        winners = []
        for _ in range(num_winners):
            candidates = random.sample(population, min(self.tournament_size, len(population)))
            winner = min(candidates, key=lambda x: x.fitness or float('inf'))
            winners.append(winner)
        return winners
    
    def _select_parents(self, population: List[AlgorithmIndividual]) -> Tuple[AlgorithmIndividual, AlgorithmIndividual]:
        return tuple(self._tournament_select(population, 2))
    
    def _load_problem_config(self) -> dict:
        config_path = os.path.join(self.problem_path, "problem_config.json")
        with open(config_path, "r", encoding="utf-8") as f:
            config = json.load(f)
            return config