import os
import json
import requests
from typing import Optional
from ..config.settings import DEFAULT_LLM_CONFIG

class LLMClient:
    def __init__(self, api_key: str, base_url: str, model: str = "gpt-3.5-turbo", temperature: float = 0.7):
        self.api_key = api_key
        self.base_url = base_url
        self.model = model
        self.default_temperature = temperature

        self.operator_temps = {
            "initialize": 1.0,  
            "crossover": 0.5,  
            "mutation": 0.9,    
            "verify": 0.1       
        }

    
    @classmethod
    def from_config(cls, problem_path: str):
        config_path = os.path.join(problem_path, "problem_config.json")
        with open(config_path, "r", encoding="utf-8") as f:
            config = json.load(f)

        llm_config = {
            **DEFAULT_LLM_CONFIG, 
            **config.get("llm_config", {})  
        }

        return cls(
            api_key=llm_config["api_key"],
            base_url=llm_config["base_url"],
            model=llm_config["model"],
            temperature=llm_config["temperature"]
        )
    
    def _clean_code(self, code: str) -> str:
        return code.replace('```python', '').replace('```', '').strip()
    
    def _call_llm(self, prompt: str, operator: str = None) -> Optional[str]:
        headers = {
            "Content-Type": "application/json",
            "Authorization": f"Bearer {self.api_key}"
        }
        
        temperature = (self.operator_temps.get(operator, self.default_temperature) 
                      if operator else self.default_temperature)
        
        data = {
            "model": self.model,
            "messages": [{"role": "user", "content": prompt}],
            "temperature": temperature
        }
        
        try:
            response = requests.post(self.base_url, headers=headers, json=data)
            response.raise_for_status()
            code = response.json()["choices"][0]["message"]["content"].strip()
            return self._clean_code(code)
        except Exception as e:
            print(str(e))
            return None