from typing import Dict, List, Tuple
from lead.core.multi_objective.individual import MultiObjectiveIndividual

class CrossoverAnalysisOperator:
    
    def analyze(self, parent1: MultiObjectiveIndividual,
               parent2: MultiObjectiveIndividual,
               objective_names: List[str]) -> Dict:
        result = {
            "dominance": self._check_dominance(parent1, parent2),
            "target_objective": None,
            "parent1_strengths": [],
            "parent2_strengths": [],
            "function_name": self._extract_function_name(parent1.code)
        }
        
        for obj in objective_names:
            if parent1.fitnesses[obj] < parent2.fitnesses[obj]: 
                result["parent1_strengths"].append(obj)
            elif parent1.fitnesses[obj] > parent2.fitnesses[obj]:
                result["parent2_strengths"].append(obj)
        
        if result["dominance"] == "parent1":
            worst_obj = max(parent2.fitnesses.items(), key=lambda x: x[1])[0]
            result["target_objective"] = worst_obj
        elif result["dominance"] == "parent2":
            worst_obj = max(parent1.fitnesses.items(), key=lambda x: x[1])[0]
            result["target_objective"] = worst_obj
        else:
            max_diff = 0
            for obj in objective_names:
                diff = abs(parent1.fitnesses[obj] - parent2.fitnesses[obj])
                if diff > max_diff:
                    max_diff = diff
                    result["target_objective"] = obj
        
        return result
    
    def _check_dominance(self, ind1: MultiObjectiveIndividual,
                        ind2: MultiObjectiveIndividual) -> str:
        if ind1.dominates(ind2):
            return "parent1"
        elif ind2.dominates(ind1):
            return "parent2"
        return "non_dominated"
        
    def _extract_function_name(self, code: str) -> str:
        for line in code.split('\n'):
            if line.strip().startswith('def '):
                return line.split()[1].split('(')[0]
        return "solution" 