from typing import Tuple, List, Dict
from lead.core.multi_objective.individual import MultiObjectiveIndividual
from .crossover_analysis import CrossoverAnalysisOperator
from ..llm_integration import LLMClient
from .verify_operator import VerifyOperator

class MultiObjectiveCrossoverOperator:
    def __init__(self, llm_client: LLMClient, verify_operator: VerifyOperator):
        self.llm_client = llm_client
        self.analysis_operator = CrossoverAnalysisOperator()
        self.verify_operator = verify_operator
    
    def crossover(self, parent1: MultiObjectiveIndividual,
                 parent2: MultiObjectiveIndividual,
                 objective_names: List[str]) -> Tuple[str, Dict]:
        analysis_info = self.analysis_operator.analyze(
            parent1, parent2, objective_names)
        prompt = self._build_multi_obj_prompt(
            parent1, parent2, analysis_info)
        new_code = self.llm_client._call_llm(prompt, operator="crossover")
        verified_code = self._verify_code(new_code, analysis_info["function_name"])
        
        return verified_code, analysis_info
    
    def _build_multi_obj_prompt(self,
                              parent1: MultiObjectiveIndividual,
                              parent2: MultiObjectiveIndividual,
                              analysis_info: Dict) -> str:
        target_obj = analysis_info["target_objective"]
        return f"""Please perform a multi-objective optimization crossover based on the following two parent algorithms, with a special focus on the objective '{target_obj}':

        Parent 1 Strengths: {', '.join(analysis_info["parent1_strengths"])}
        Parent 1 Code:
        {parent1.code}

        Parent 2 Strengths: {', '.join(analysis_info["parent2_strengths"])}
        Parent 2 Code:
        {parent2.code}

        Please generate a new algorithm with the following requirements:
        1. Preserve the advantages of each parent on their respective strength objectives.
        2. Specifically optimize performance for the '{target_obj}' objective.
        3. Ensure the code is complete and meets the problem's requirements.
        4. The function name must be: {analysis_info["function_name"]}

        Return only the final Python code, without any explanation.
        """
    
    def _verify_code(self, code: str, function_name: str) -> str:
        return self.verify_operator.verify_code_format(code, function_name)