from typing import Tuple, List, Dict
from lead.core.multi_objective.individual import MultiObjectiveIndividual
from .mutation_analysis import MutationAnalysisOperator
from ..llm_integration import LLMClient
from .verify_operator import VerifyOperator

class MultiObjectiveMutationOperator:
    def __init__(self, llm_client: LLMClient, verify_operator: VerifyOperator):
        self.llm_client = llm_client
        self.analysis_operator = MutationAnalysisOperator()
        self.verify_operator = verify_operator
    
    def mutate(self, individual: MultiObjectiveIndividual,
              objective_names: List[str],
              reference_front: List[Dict] = None) -> Tuple[str, Dict]:
        analysis_info = self.analysis_operator.analyze(
            individual, objective_names, reference_front)

        if not analysis_info["target_objective"]:
            analysis_info["target_objective"] = objective_names[0] if objective_names else "error"

        prompt = self._build_multi_obj_prompt(
            individual, analysis_info)

        new_code = self.llm_client._call_llm(prompt, operator="mutation")

        verified_code = self._verify_code(
            new_code, analysis_info["function_name"])
        
        return verified_code, analysis_info
    
    def _build_multi_obj_prompt(self,
                              individual: MultiObjectiveIndividual,
                              analysis_info: Dict) -> str:
        target_obj = analysis_info["target_objective"]
        improvement = analysis_info["improvement_potential"].get(target_obj, 1.0)
        
        return f"""Please perform a multi-objective optimization mutation on the following algorithm, focusing on improving the objective '{target_obj}' (current improvement potential: {improvement:.2f}):

        Original Code:
        {individual.code}

        Please:
        1. Maintain performance on the other objectives.
        2. Specifically optimize the '{target_obj}' objective.
        3. You may introduce new mathematical concepts or optimization methods.
        4. Ensure the code is complete and executable.
        5. Avoid using recursion or limit its depth.
        6. The function name must remain: {analysis_info["function_name"]}

        Return only the final Python code, without any explanation.
        """
    
    def _verify_code(self, code: str, function_name: str) -> str:
        return self.verify_operator.verify_code_format(code, function_name)