import os
import json
import importlib.util
import traceback
from typing import Dict, List, Any
from ..data_loader import DataLoader
from ..timeout import Timeout, TimeoutError
from lead.core.multi_objective.individual import MultiObjectiveIndividual

class MultiObjectiveEvaluator:

    def __init__(self, problem_path: str):
        self.problem_path = problem_path
        self.evaluation_module = self._load_evaluation_module()
        self.data_loader = self._load_data_loader()
        self.objective_names = self._load_objective_names()

        if self.data_loader:
            self.dataset = self.data_loader.get_data()
        else:
            self.dataset = None

        config = self._load_problem_config()
        self.timeout = config.get("evaluation_timeout", 30)
    
    def evaluate(self, code: str) -> Dict[str, float]:
        try:
            namespace = {}
            exec(code, namespace)
            config = self._load_problem_config()
            func = namespace[config["function_name"]]
            timeout = Timeout(self.timeout)
            # print(f"[DEBUG] Dataset structure: {self.dataset}")
            if self.dataset is not None:
                results = timeout.run(self.evaluation_module.evaluate, func, self.dataset)
            else:
                results = timeout.run(self.evaluation_module.evaluate, func)
            if isinstance(results, dict):
                return {name: results[name] for name in self.objective_names}
            elif isinstance(results, (list, tuple)):
                return {name: results[i] for i, name in enumerate(self.objective_names)}
            else:
                raise ValueError("The evaluation function should return a dictionary or a list/tuple.")
                
        except TimeoutError:
            return {name: float('inf') for name in self.objective_names}
        except Exception as e:
            return {name: float('inf') for name in self.objective_names}
    
    def _load_objective_names(self) -> List[str]:
        config = self._load_problem_config()
        return config.get("objective_names", ["f1", "f2"])
    
    def _load_evaluation_module(self):
        eval_path = os.path.join(self.problem_path, "evaluation.py")
        spec = importlib.util.spec_from_file_location("evaluation", eval_path)
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)
        return module
    
    def _load_data_loader(self):
        loader_path = os.path.join(self.problem_path, "dataset", "data_loader.py")
        if not os.path.exists(loader_path):
            return None
            
        spec = importlib.util.spec_from_file_location("data_loader", loader_path)
        module = importlib.util.module_from_spec(spec)
        spec.loader.exec_module(module)
        return module.ProblemDataLoader(self.problem_path)
    
    def _load_problem_config(self) -> dict:
        config_path = os.path.join(self.problem_path, "problem_config.json")
        with open(config_path, "r", encoding="utf-8") as f:
            return json.load(f)