import os
import json
from datetime import datetime
from typing import List, Dict, Any
from ..core.individual import AlgorithmIndividual

class GenerationStorage:
    def __init__(self, problem_path: str):
        self.problem_path = problem_path
        self.storage_dir = os.path.join(problem_path, "evolution_history", 
                                      datetime.now().strftime("%Y%m%d_%H%M%S"))
        os.makedirs(self.storage_dir, exist_ok=True)

        self.log_file = os.path.join(problem_path, "evolution.log")
        
    def save_generation(self, generation: int, data: dict):
        file_path = os.path.join(self.storage_dir, f"generation_{generation}.json")
        with open(file_path, "w", encoding="utf-8") as f:
            json.dump(data, f, indent=2, ensure_ascii=False)
    
    def load_generation(self, generation: int) -> dict:
        file_path = os.path.join(self.storage_dir, f"generation_{generation}.json")
        if not os.path.exists(file_path):
            return None
        with open(file_path, "r", encoding="utf-8") as f:
            return json.load(f)
    
    def get_best_individual(self) -> dict:
        best_fitness = float('inf')
        best_individual = None
        
        for file in os.listdir(self.storage_dir):
            if file.startswith("generation_") and file.endswith(".json"):
                with open(os.path.join(self.storage_dir, file), "r", encoding="utf-8") as f:
                    data = json.load(f)
                    if data["best_ever_fitness"] < best_fitness:
                        best_fitness = data["best_ever_fitness"]
                        best_individual = {
                            "code": data["best_ever_code"],
                            "fitness": data["best_ever_fitness"],
                            "generation": int(file.split("_")[1].split(".")[0])
                        }
        
        return best_individual
    
    def _individual_to_dict(self, individual: AlgorithmIndividual) -> Dict[str, Any]:
        return {
            "code": individual.code,
            "generation": individual.generation,
            "fitness": individual.fitness
        }
    
    def _dict_to_individual(self, data: Dict[str, Any]) -> AlgorithmIndividual:
        return AlgorithmIndividual(
            code=data["code"],
            generation=data["generation"],
            fitness=data["fitness"]
        )
    
    def _log_generation(self, generation: int, best_individual: AlgorithmIndividual):
        log_entry = (
            f"Generation {generation:04d} | "
            f"Best Fitness: {best_individual.fitness:.6f} | "
            f"Timestamp: {datetime.now().isoformat()}\n"
        )
        with open(self.log_file, "a", encoding="utf-8") as f:
            f.write(log_entry) 