from typing import List
import random
import numpy as np
from .base import BaseEvolutionAlgorithm
from ..individual import AlgorithmIndividual
from ..llm_integration import LLMClient
from ..operators import CrossoverOperator, MutationOperator

class TournamentEvolution(BaseEvolutionAlgorithm):
    def __init__(self, config: dict, llm_client: LLMClient):
        super().__init__(config)
        self.tournament_size = config.get("tournament_size", 3)
        self.llm_client = llm_client
        self.crossover_operator = CrossoverOperator(llm_client)
        self.mutation_operator = MutationOperator(llm_client)
    
    def select(self, population: List[AlgorithmIndividual], num_parents: int) -> List[AlgorithmIndividual]:
        select_size = max(2, int(np.ceil(len(population) * 0.5)))
        winners = []
        for _ in range(select_size):
            candidates = random.sample(population, min(self.tournament_size, len(population)))
            winner = min(candidates, key=lambda x: x.fitness or float('inf'))
            winners.append(winner)
        return winners
    
    def crossover(self, parents: List[AlgorithmIndividual]) -> List[AlgorithmIndividual]:
        if len(parents) < 2:
            return parents
        
        if random.random() < self.config["crossover_rate"]:
            child_code = self.crossover_operator.crossover(parents[0].code, parents[1].code)
            if child_code:
                return [AlgorithmIndividual(child_code, parents[0].generation + 1)]
        return [parents[0]]
    
    def mutate(self, individual: AlgorithmIndividual) -> AlgorithmIndividual:
        if random.random() < self.config["mutation_rate"]:
            mutated_code = self.mutation_operator.mutate(individual.code)
            if mutated_code:
                return AlgorithmIndividual(mutated_code, individual.generation)
        return individual
    
    def survive(self, population: List[AlgorithmIndividual], offspring: List[AlgorithmIndividual], 
               pop_size: int) -> List[AlgorithmIndividual]:
        return offspring[:pop_size]