from dataclasses import dataclass, field
from typing import Dict, Any, List
import numpy as np

@dataclass
class MultiObjectiveIndividual:
    code: str
    generation: int
    fitnesses: Dict[str, float] = field(default_factory=dict)
    metadata: Dict[str, Any] = field(default_factory=dict)
    
    def dominates(self, other: 'MultiObjectiveIndividual') -> bool:
        if not self.fitnesses or not other.fitnesses:
            return False
            
        at_least_one_better = False
        for key in self.fitnesses:
            if self.fitnesses[key] > other.fitnesses[key]:  
                return False
            if self.fitnesses[key] < other.fitnesses[key]:
                at_least_one_better = True
        return at_least_one_better
    
    def to_dict(self) -> Dict[str, Any]:
        return {
            "code": self.code,
            "generation": self.generation,
            "fitnesses": self.fitnesses,
            "metadata": self.metadata
        }
    
    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'MultiObjectiveIndividual':
        return cls(
            code=data["code"],
            generation=data["generation"],
            fitnesses=data.get("fitnesses", {}),
            metadata=data.get("metadata", {})
        )
    
    def crowding_distance(self, front: List['MultiObjectiveIndividual']) -> float:
        if not front or len(front) == 1:
            return float('inf')
            
        distance = 0.0
        for obj in self.fitnesses:
            sorted_front = sorted(front, key=lambda x: x.fitnesses[obj])
            min_val = sorted_front[0].fitnesses[obj]
            max_val = sorted_front[-1].fitnesses[obj]
            
            for i, ind in enumerate(sorted_front):
                if ind == self:
                    if i == 0 or i == len(sorted_front) - 1:
                        distance += float('inf')
                    else:
                        distance += (sorted_front[i+1].fitnesses[obj] - 
                                   sorted_front[i-1].fitnesses[obj]) / (max_val - min_val)
                    break
        return distance