from typing import Optional, List
from ..llm_integration import LLMClient
from .verify_operator import VerifyOperator

class AlgorithmIdeasGenerator:
    def __init__(self, llm_client: LLMClient):
        self.llm_client = llm_client
        
    def generate_ideas(self, problem_desc: str, num_ideas: int) -> List[str]:
        prompt = f"""Please generate {num_ideas} different algorithmic approaches for the following problem:

Problem Description: {problem_desc}

Requirements:
1. Describe each approach in a single, concise line.
2. Do not include any other information.
3. The approaches should be distinctly different from one another.
4. Return exactly {num_ideas} lines in total.

Example Format:
Approach 1 description
Approach 2 description
...
Approach {num_ideas} description
"""
        response = self.llm_client._call_llm(prompt, operator="generate_ideas")
        return [line.strip() for line in response.split('\n') if line.strip()]

class InitializeOperator:
    def __init__(self, llm_client: LLMClient):
        self.llm_client = llm_client
        self.ideas_generator = AlgorithmIdeasGenerator(llm_client)
        self.verify_operator = VerifyOperator(llm_client)

    def generate_initial_code(self, problem_desc: str, function_name: str,
                            input_fmt: str, output_fmt: str, idea: str = None) -> str:

        base_prompt = f"""Please provide a Python code implementation for the described problem using the following algorithmic idea:

Problem Description: {problem_desc}
Idea: {idea or 'No specific idea'}

Specific Requirements:
1. Function Name: {function_name}
2. Input: {input_fmt}
3. Return Value: {output_fmt}

Note:
- No type annotations are needed.
- Return only the function code; do not include any explanations or test cases.
- The code can include multiple helper functions.
- The code should be concise and efficient.
"""
        code = self.llm_client._call_llm(base_prompt, operator="initialize")
        if code:
            return self.verify_operator.verify_code_format(code, function_name)
        return None