import json
import random
from collections import deque
import math

def strategy_foraging_1(obs):

    def is_valid_move(view, pos):
        i, j = pos
        if 0 <= i < len(view) and 0 <= j < len(view[0]):
            cell = view[i][j]
            return cell == "." or cell == "F" or cell == "N"
        return False
    obs = json.loads(obs)
    carrying_food = obs.get("carrying_food", False)
    view = obs["views"][-1]  # Get current view
    agent_pos = [len(view) // 2, len(view[0]) // 2]  # Agent is at the center of view

    # Search for food source and nest positions in the view
    food_positions = []
    nest_positions = []

    for i in range(len(view)):
        for j in range(len(view[i])):
            if view[i][j] == "F":
                food_positions.append((i, j))
            elif view[i][j] == "N":
                nest_positions.append((i, j))

    # Determine target based on whether agent is carrying food
    if carrying_food:
        target_positions = nest_positions
    else:
        target_positions = food_positions

    # If target is found in view
    if target_positions:
        # Sort by Manhattan distance
        target_positions.sort(key=lambda pos: abs(pos[0] - agent_pos[0]) + abs(pos[1] - agent_pos[1]))
        target_pos = target_positions[0]

        # Calculate direction to move
        dy = target_pos[0] - agent_pos[0]
        dx = target_pos[1] - agent_pos[1]

        # If adjacent to target, stay
        if abs(dx) + abs(dy) == 1:
            return "ACTION: STAY"

        # Prioritize vertical movement
        if dy < 0:
            next_pos = (agent_pos[0] - 1, agent_pos[1])
            if is_valid_move(view, next_pos):
                return "ACTION: UP"
        elif dy > 0:
            next_pos = (agent_pos[0] + 1, agent_pos[1])
            if is_valid_move(view, next_pos):
                return "ACTION: DOWN"

        # Then try horizontal movement
        if dx < 0:
            next_pos = (agent_pos[0], agent_pos[1] - 1)
            if is_valid_move(view, next_pos):
                return "ACTION: LEFT"
        elif dx > 0:
            next_pos = (agent_pos[0], agent_pos[1] + 1)
            if is_valid_move(view, next_pos):
                return "ACTION: RIGHT"

    # If no clear path or target not in view, try random valid direction
    directions = [("UP", (-1, 0)), ("DOWN", (1, 0)), ("LEFT", (0, -1)), ("RIGHT", (0, 1))]
    import random
    random.shuffle(directions)

    for action, (dy, dx) in directions:
        next_pos = (agent_pos[0] + dy, agent_pos[1] + dx)
        if is_valid_move(view, next_pos):
            return f"ACTION: {action}"

    return "ACTION: STAY"


def strategy_foraging_2(obs):

    def get_direction(y1, x1, y2, x2):
        if y1 > y2: return "UP"
        if y1 < y2: return "DOWN"
        if x1 > x2: return "LEFT"
        if x1 < x2: return "RIGHT"
        return None

    def bfs_to_target(view, target_type):
        h, w = len(view), len(view[0])
        start = (h // 2, w // 2)  # Agent is at center of view

        # BFS to find the shortest path
        visited = set([start])
        queue = deque([(start, [])])  # (position, path)

        while queue:
            (y, x), path = queue.popleft()

            # Check if we've reached the target
            if view[y][x] == target_type:
                if not path:  # Already at target
                    return "STAY"
                return path[0]  # Return first step in path

            # Adjacent to target, stay put
            for ny, nx in [(y - 1, x), (y + 1, x), (y, x - 1), (y, x + 1)]:
                if 0 <= ny < h and 0 <= nx < w and view[ny][nx] == target_type:
                    direction = get_direction(y, x, ny, nx)
                    return "STAY" if not direction else direction

            # Try each direction
            directions = [
                (y - 1, x, "UP"),
                (y + 1, x, "DOWN"),
                (y, x - 1, "LEFT"),
                (y, x + 1, "RIGHT")
            ]

            for ny, nx, direction in directions:
                if 0 <= ny < h and 0 <= nx < w and (ny, nx) not in visited:
                    cell = view[ny][nx]
                    if cell == "." or cell == target_type:  # Can move to empty space or target
                        visited.add((ny, nx))
                        queue.append(((ny, nx), path + [direction]))

        # If no path is found, try any valid move
        directions = ["UP", "DOWN", "LEFT", "RIGHT"]
        for direction in directions:
            dy, dx = 0, 0
            if direction == "UP":
                dy = -1
            elif direction == "DOWN":
                dy = 1
            elif direction == "LEFT":
                dx = -1
            elif direction == "RIGHT":
                dx = 1

            ny, nx = start[0] + dy, start[1] + dx
            if 0 <= ny < h and 0 <= nx < w:
                cell = view[ny][nx]
                if cell == "." or cell == target_type:
                    return direction

        return "STAY"

    obs = json.loads(obs)
    carrying_food = obs.get("carrying_food", False)
    view = obs["views"][-1]  # Get current view

    # Find target in the view (Food or Nest)
    target_type = "N" if carrying_food else "F"

    # Find the shortest path to the target using BFS
    action = bfs_to_target(view, target_type)

    return f"ACTION: {action}"


def strategy_foraging_3(obs):
    def is_valid_move(view, direction):
        h, w = len(view), len(view[0])
        y, x = h // 2, w // 2

        if direction == "UP":
            y -= 1
        elif direction == "DOWN":
            y += 1
        elif direction == "LEFT":
            x -= 1
        elif direction == "RIGHT":
            x += 1

        if 0 <= y < h and 0 <= x < w:
            cell = view[y][x]
            return cell == "." or cell == "F" or cell == "N"
        return False

    def random_valid_move(view):
        import random
        directions = ["UP", "DOWN", "LEFT", "RIGHT"]
        random.shuffle(directions)

        for direction in directions:
            if is_valid_move(view, direction):
                return direction

        return "STAY"

    def bfs_to_coordinates(view, target):
        h, w = len(view), len(view[0])
        start = (h // 2, w // 2)  # Agent position
        target = (target[0], target[1])  # Target position

        # Check if already at target
        if start == target:
            return "STAY"

        # Check if adjacent to target
        if (abs(start[0] - target[0]) + abs(start[1] - target[1])) == 1:
            return "STAY"  # For pickup/dropoff

        # BFS to find path
        queue = deque([(start, [])])  # (position, path)
        visited = set([start])

        while queue:
            (y, x), path = queue.popleft()

            if (y, x) == target:
                return path[0] if path else "STAY"

            for dy, dx, direction in [(-1, 0, "UP"), (1, 0, "DOWN"), (0, -1, "LEFT"), (0, 1, "RIGHT")]:
                ny, nx = y + dy, x + dx
                if 0 <= ny < h and 0 <= nx < w and (ny, nx) not in visited:
                    cell = view[ny][nx]
                    if cell == "." or cell == "F" or cell == "N" or (ny, nx) == target:
                        visited.add((ny, nx))
                        queue.append(((ny, nx), path + [direction]))

        # If no path found, move randomly
        return random_valid_move(view)

    obs = json.loads(obs)
    carrying_food = obs.get("carrying_food", False)
    pos = obs["pos"]
    view = obs["views"][-1]
    messages = obs.get("messages", [])

    # Process received messages to get known locations
    food_pos = None
    nest_pos = None

    for msg in messages:
        if msg.startswith("LOC:"):
            parts = msg.split(":")
            if len(parts) >= 3:
                loc_type = parts[1]
                try:
                    coords = [int(c) for c in parts[2].split(",")]
                    if loc_type == "F":
                        food_pos = coords
                    elif loc_type == "N":
                        nest_pos = coords
                except:
                    pass

    # Scan view to find food and nest
    view_food_pos = None
    view_nest_pos = None
    for i in range(len(view)):
        for j in range(len(view[i])):
            if view[i][j] == "F":
                # Convert to global coordinates
                view_food_pos = [pos[0] - (len(view) // 2) + i, pos[1] - (len(view[0]) // 2) + j]
            elif view[i][j] == "N":
                view_nest_pos = [pos[0] - (len(view) // 2) + i, pos[1] - (len(view[0]) // 2) + j]

    # Update known positions and broadcast them
    message = ""
    if view_food_pos and (not food_pos or view_food_pos != food_pos):
        food_pos = view_food_pos
        message = f"LOC:F:{food_pos[0]},{food_pos[1]}"
    elif view_nest_pos and (not nest_pos or view_nest_pos != nest_pos):
        nest_pos = view_nest_pos
        message = f"LOC:N:{nest_pos[0]},{nest_pos[1]}"

    # Determine target based on carrying status
    target_type = "N" if carrying_food else "F"
    target_pos = nest_pos if carrying_food else food_pos

    # Find path using BFS
    action = "STAY"
    if target_pos:
        # Convert target to local view coordinates
        local_target = [
            target_pos[0] - pos[0] + (len(view) // 2),
            target_pos[1] - pos[1] + (len(view[0]) // 2)
        ]

        # Check if target is in view bounds
        if 0 <= local_target[0] < len(view) and 0 <= local_target[1] < len(view[0]):
            action = bfs_to_coordinates(view, local_target)
        else:
            # Move in general direction of target
            dy = target_pos[0] - pos[0]
            dx = target_pos[1] - pos[1]

            # Prioritize the direction with larger magnitude
            if abs(dy) > abs(dx):
                action = "UP" if dy < 0 else "DOWN"
                if not is_valid_move(view, action):
                    action = "LEFT" if dx < 0 else "RIGHT"
                    if not is_valid_move(view, action):
                        action = random_valid_move(view)
            else:
                action = "LEFT" if dx < 0 else "RIGHT"
                if not is_valid_move(view, action):
                    action = "UP" if dy < 0 else "DOWN"
                    if not is_valid_move(view, action):
                        action = random_valid_move(view)
    else:
        # If target position is unknown, explore
        action = random_valid_move(view)

    if message:
        return f"MSG: {message}\nACTION: {action}"
    else:
        return f"ACTION: {action}"


def strategy_transport_1(obs):
    # Parse observation
    obs = json.loads(obs)
    view = obs["views"][-1]  # Current view
    pos = obs["pos"]  # Current position
    center = len(view) // 2  # Center position in view

    # Find obstacle and other agents in view
    obstacle_positions = []
    agent_positions = []
    exit_positions = []

    for i in range(len(view)):
        for j in range(len(view[i])):
            if view[i][j] == "B":
                obstacle_positions.append((i, j))
            elif view[i][j] not in [".", "W", "*"]:
                agent_positions.append((i, j))
            elif view[i][j] == "*":
                exit_positions.append((i, j))

    # Count agents near obstacles
    pushing_agents = {}
    for obs_pos in obstacle_positions:
        pushing_agents[obs_pos] = 0
        for agent_pos in agent_positions:
            if abs(agent_pos[0] - obs_pos[0]) <= 1 and abs(agent_pos[1] - obs_pos[1]) <= 1:
                pushing_agents[obs_pos] += 1

    # Parse messages to get information about obstacle positions from other agents
    messages = obs.get("messages", [])
    for msg in messages:
        if msg.startswith("OBS:"):
            try:
                parts = msg.split(":")
                if len(parts) >= 2:
                    coords = parts[1].strip().split(",")
                    if len(coords) == 2:
                        o_i, o_j = int(coords[0]), int(coords[1])
                        if (o_i, o_j) not in obstacle_positions:
                            obstacle_positions.append((o_i, o_j))
            except:
                pass

    # State management based on obstacles and agents
    if obstacle_positions:
        # Found an obstacle, try to push it
        nearest_obstacle = min(obstacle_positions, key=lambda p: abs(p[0] - center) + abs(p[1] - center))

        # Check if enough agents are nearby for pushing
        if nearest_obstacle in pushing_agents and pushing_agents[nearest_obstacle] >= 5:
            # Push the obstacle
            di = nearest_obstacle[0] - center
            dj = nearest_obstacle[1] - center

            # Move toward obstacle if not adjacent, push if adjacent
            if abs(di) + abs(dj) > 1:
                # Move toward obstacle
                if abs(di) > abs(dj):
                    return f"MSG: OBS:{nearest_obstacle[0]},{nearest_obstacle[1]}\nACTION: {'DOWN' if di > 0 else 'UP'}"
                else:
                    return f"MSG: OBS:{nearest_obstacle[0]},{nearest_obstacle[1]}\nACTION: {'RIGHT' if dj > 0 else 'LEFT'}"
            else:
                # Push obstacle
                if di != 0:
                    return f"MSG: PUSH:{nearest_obstacle[0]},{nearest_obstacle[1]}\nACTION: {'DOWN' if di > 0 else 'UP'}"
                else:
                    return f"MSG: PUSH:{nearest_obstacle[0]},{nearest_obstacle[1]}\nACTION: {'RIGHT' if dj > 0 else 'LEFT'}"
        else:
            # Move toward obstacle to help push
            di = nearest_obstacle[0] - center
            dj = nearest_obstacle[1] - center

            if abs(di) > abs(dj):
                return f"MSG: OBS:{nearest_obstacle[0]},{nearest_obstacle[1]}\nACTION: {'DOWN' if di > 0 else 'UP'}"
            else:
                return f"MSG: OBS:{nearest_obstacle[0]},{nearest_obstacle[1]}\nACTION: {'RIGHT' if dj > 0 else 'LEFT'}"

    # If obstacle is pushed out or not found in view, look for exit
    if exit_positions:
        nearest_exit = min(exit_positions, key=lambda p: abs(p[0] - center) + abs(p[1] - center))
        di = nearest_exit[0] - center
        dj = nearest_exit[1] - center

        if abs(di) > abs(dj):
            return f"MSG: EXIT\nACTION: {'DOWN' if di > 0 else 'UP'}"
        else:
            return f"MSG: EXIT\nACTION: {'RIGHT' if dj > 0 else 'LEFT'}"

    # Explore randomly if no obstacle or exit is found
    actions = ["UP", "DOWN", "LEFT", "RIGHT"]
    return f"MSG: EXPLORE\nACTION: {random.choice(actions)}"


def strategy_transport_2(obs):
    # Parse observation
    obs = json.loads(obs)
    view = obs["views"][-1]  # Current view
    pos = obs["pos"]  # Current position
    center = len(view) // 2  # Center position in view

    # Find obstacle, walls, agents, and exits in view
    obstacle_positions = []
    wall_positions = []
    agent_positions = []
    exit_positions = []

    for i in range(len(view)):
        for j in range(len(view[i])):
            if view[i][j] == "B":
                obstacle_positions.append((i, j))
            elif view[i][j] == "W":
                wall_positions.append((i, j))
            elif view[i][j] not in [".", "W", "*"]:
                agent_positions.append((i, j))
            elif view[i][j] == "*":
                exit_positions.append((i, j))

    # Process messages from other agents
    messages = obs.get("messages", [])
    target_obstacle = None
    obstacle_agents = {}

    for msg in messages:
        if msg.startswith("TARGET:"):
            try:
                parts = msg.split(":")
                if len(parts) >= 2:
                    coords = parts[1].strip().split(",")
                    if len(coords) == 2:
                        o_i, o_j = int(coords[0]), int(coords[1])
                        target_obstacle = (o_i, o_j)
                    if len(parts) >= 3:
                        count = int(parts[2])
                        obstacle_agents[(o_i, o_j)] = count
            except:
                pass

    # Phase determination: Find obstacle, coordinate pushing, or escape
    phase = "FIND"

    # Check if obstacles have been pushed out (escape phase)
    if len(obstacle_positions) == 0 and (target_obstacle is None or obs.get("history", [])):
        phase = "ESCAPE"
    elif len(obstacle_positions) > 0:
        phase = "PUSH"

    # Strategy based on the phase
    if phase == "PUSH":
        # If there's a target obstacle from messages, prioritize it
        if target_obstacle and target_obstacle in obstacle_positions:
            nearest_obstacle = target_obstacle
        else:
            # Find the nearest obstacle
            nearest_obstacle = min(obstacle_positions, key=lambda p: abs(p[0] - center) + abs(p[1] - center))

        # Count agents around this obstacle
        agent_count = 0
        for agent_pos in agent_positions:
            if abs(agent_pos[0] - nearest_obstacle[0]) <= 1 and abs(agent_pos[1] - nearest_obstacle[1]) <= 1:
                agent_count += 1

        # Update obstacle agent count from our observation
        obstacle_agents[nearest_obstacle] = agent_count

        # Check if enough agents are around to push
        can_push = agent_count >= 5

        # Calculate direction to obstacle
        di = nearest_obstacle[0] - center
        dj = nearest_obstacle[1] - center

        # Agent is adjacent to obstacle
        if abs(di) + abs(dj) == 1:
            if can_push:
                # Push the obstacle
                if di == 1:
                    return f"MSG: TARGET:{nearest_obstacle[0]},{nearest_obstacle[1]}:{agent_count}\nACTION: DOWN"
                elif di == -1:
                    return f"MSG: TARGET:{nearest_obstacle[0]},{nearest_obstacle[1]}:{agent_count}\nACTION: UP"
                elif dj == 1:
                    return f"MSG: TARGET:{nearest_obstacle[0]},{nearest_obstacle[1]}:{agent_count}\nACTION: RIGHT"
                else:  # dj == -1
                    return f"MSG: TARGET:{nearest_obstacle[0]},{nearest_obstacle[1]}:{agent_count}\nACTION: LEFT"
            else:
                # Stay adjacent to the obstacle and wait for more agents
                return f"MSG: TARGET:{nearest_obstacle[0]},{nearest_obstacle[1]}:{agent_count}\nACTION: STAY"
        else:
            # Move toward the obstacle
            if abs(di) > abs(dj):
                return f"MSG: TARGET:{nearest_obstacle[0]},{nearest_obstacle[1]}:{agent_count}\nACTION: {'DOWN' if di > 0 else 'UP'}"
            else:
                return f"MSG: TARGET:{nearest_obstacle[0]},{nearest_obstacle[1]}:{agent_count}\nACTION: {'RIGHT' if dj > 0 else 'LEFT'}"

    elif phase == "ESCAPE":
        # Move toward the nearest exit
        if exit_positions:
            nearest_exit = min(exit_positions, key=lambda p: abs(p[0] - center) + abs(p[1] - center))
            di = nearest_exit[0] - center
            dj = nearest_exit[1] - center

            # Choose the shortest path to exit
            if abs(di) > abs(dj):
                return f"MSG: ESCAPE\nACTION: {'DOWN' if di > 0 else 'UP'}"
            else:
                return f"MSG: ESCAPE\nACTION: {'RIGHT' if dj > 0 else 'LEFT'}"

        # If no exit is visible, move away from walls
        if wall_positions:
            # Calculate average wall position
            avg_wall_i = sum(w[0] for w in wall_positions) / len(wall_positions)
            avg_wall_j = sum(w[1] for w in wall_positions) / len(wall_positions)

            # Move away from average wall position
            di = center - avg_wall_i
            dj = center - avg_wall_j

            if abs(di) > abs(dj):
                return f"MSG: ESCAPE\nACTION: {'DOWN' if di > 0 else 'UP'}"
            else:
                return f"MSG: ESCAPE\nACTION: {'RIGHT' if dj > 0 else 'LEFT'}"

    # Default behavior: explore randomly
    actions = ["UP", "DOWN", "LEFT", "RIGHT"]
    return f"MSG: FIND\nACTION: {random.choice(actions)}"


def strategy_transport_3(obs):
    # Parse observation
    obs = json.loads(obs)
    view = obs["views"][-1]  # Current view
    history = obs.get("history", [])
    center = len(view) // 2  # Center position in view

    # Map analysis
    obstacle_positions = []
    agent_positions = []
    exit_positions = []

    for i in range(len(view)):
        for j in range(len(view[i])):
            if view[i][j] == "B":
                obstacle_positions.append((i, j))
            elif view[i][j] not in [".", "W", "*"]:
                agent_positions.append((i, j))
            elif view[i][j] == "*":
                exit_positions.append((i, j))

    # Parse messages from other agents
    messages = obs.get("messages", [])
    known_obstacles = {}  # Dictionary to track obstacles and pushing agents

    for msg in messages:
        try:
            if msg.startswith("OBS:"):
                parts = msg.split(":")
                if len(parts) >= 3:
                    coords = parts[1].strip().split(",")
                    agents = int(parts[2])
                    if len(coords) == 2:
                        o_i, o_j = int(coords[0]), int(coords[1])
                        known_obstacles[(o_i, o_j)] = agents
            elif msg.startswith("ROLE:"):
                # Future expansion for role-based coordination
                pass
        except:
            pass

    # Determine role based on position and game state
    role = "EXPLORER"

    # Update obstacle information from view
    for obs_pos in obstacle_positions:
        count = 0
        for agent_pos in agent_positions:
            if abs(agent_pos[0] - obs_pos[0]) <= 1 and abs(agent_pos[1] - obs_pos[1]) <= 1:
                count += 1
        known_obstacles[obs_pos] = count

    # Check if any obstacle has enough agents to push
    push_target = None
    for obs_pos, count in known_obstacles.items():
        if count >= 4:  # We need 5 agents total, and this agent counts as one
            push_target = obs_pos
            break

    # If there's a push target, become a PUSHER
    if push_target and push_target in obstacle_positions:
        role = "PUSHER"
    # If there are obstacles in view but not enough agents, become a GATHERER
    elif obstacle_positions:
        role = "GATHERER"
    # If no obstacles but exits visible, become an ESCAPER
    elif exit_positions and not obstacle_positions:
        role = "ESCAPER"

    # Act based on the determined role
    if role == "PUSHER":
        # Push the obstacle with sufficient agents
        obs_i, obs_j = push_target
        di = obs_i - center
        dj = obs_j - center

        # If adjacent to the obstacle, push it
        if abs(di) + abs(dj) == 1:
            if di == 1:
                direction = "DOWN"
            elif di == -1:
                direction = "UP"
            elif dj == 1:
                direction = "RIGHT"
            else:  # dj == -1
                direction = "LEFT"
            return f"MSG: OBS:{obs_i},{obs_j}:{known_obstacles[push_target]}\nACTION: {direction}"
        # Otherwise, move toward it
        else:
            if abs(di) > abs(dj):
                direction = "DOWN" if di > 0 else "UP"
            else:
                direction = "RIGHT" if dj > 0 else "LEFT"
            return f"MSG: OBS:{obs_i},{obs_j}:{known_obstacles[push_target]}\nACTION: {direction}"

    elif role == "GATHERER":
        # Move toward the nearest obstacle to help push
        nearest_obs = min(obstacle_positions, key=lambda p: abs(p[0] - center) + abs(p[1] - center))
        obs_i, obs_j = nearest_obs
        di = obs_i - center
        dj = obs_j - center

        if abs(di) > abs(dj):
            direction = "DOWN" if di > 0 else "UP"
        else:
            direction = "RIGHT" if dj > 0 else "LEFT"

        # Count agents near this obstacle
        agent_count = 0
        for agent_pos in agent_positions:
            if abs(agent_pos[0] - obs_i) <= 1 and abs(agent_pos[1] - obs_j) <= 1:
                agent_count += 1

        return f"MSG: OBS:{obs_i},{obs_j}:{agent_count}\nACTION: {direction}"

    elif role == "ESCAPER":
        # Move toward the nearest exit
        if exit_positions:
            nearest_exit = min(exit_positions, key=lambda p: abs(p[0] - center) + abs(p[1] - center))
            di = nearest_exit[0] - center
            dj = nearest_exit[1] - center

            if abs(di) > abs(dj):
                direction = "DOWN" if di > 0 else "UP"
            else:
                direction = "RIGHT" if dj > 0 else "LEFT"
            return f"MSG: ROLE:ESCAPER\nACTION: {direction}"

    # Default EXPLORER behavior
    # Use simple exploration pattern: if we've been moving in one direction, keep going
    if history:
        last_action = history[-1]["action"]
        # Slight preference to continue in same direction to avoid getting stuck
        if random.random() < 0.7 and last_action in ["UP", "DOWN", "LEFT", "RIGHT"]:
            return f"MSG: ROLE:EXPLORER\nACTION: {last_action}"

    # Otherwise random movement
    directions = ["UP", "DOWN", "LEFT", "RIGHT"]
    return f"MSG: ROLE:EXPLORER\nACTION: {random.choice(directions)}"


def strategy_flocking_1(obs):
    obs = json.loads(obs)

    # Current position and view
    pos = obs["pos"]
    current_view = obs["views"][-1]
    target = obs["target"]

    # Get agent ID (which is at the center of the view)
    view_center = len(current_view) // 2
    agent_id = int(current_view[view_center][view_center])

    # Determine the target position in the target shape
    # Get positions where 'A' appears in the target
    target_positions = []
    for i in range(len(target)):
        for j in range(len(target[i])):
            if target[i][j] == 'A':
                target_positions.append([i, j])

    # Assign a position to this agent based on its ID
    # Using modulo to handle cases where there are more agents than target positions
    assigned_pos = target_positions[agent_id % len(target_positions)]

    # Determine the direction to move toward the assigned position
    # For simplicity, we'll greedily move toward the target position
    di = assigned_pos[0] - pos[0]
    dj = assigned_pos[1] - pos[1]

    # Choose action based on the largest difference
    action = "STAY"
    if abs(di) > abs(dj):
        if di > 0:
            action = "DOWN"  # Move down to increase row index
        elif di < 0:
            action = "UP"  # Move up to decrease row index
    else:
        if dj > 0:
            action = "RIGHT"  # Move right to increase column index
        elif dj < 0:
            action = "LEFT"  # Move left to decrease column index

    # Check if the chosen action would hit a wall
    new_i, new_j = pos[0], pos[1]
    if action == "UP":
        new_i -= 1
    elif action == "DOWN":
        new_i += 1
    elif action == "LEFT":
        new_j -= 1
    elif action == "RIGHT":
        new_j += 1

    # Get coordinates in the view
    view_i = view_center + (new_i - pos[0])
    view_j = view_center + (new_j - pos[1])

    # Check if the new position is valid
    if (0 <= view_i < len(current_view) and
            0 <= view_j < len(current_view[0]) and
            current_view[view_i][view_j] not in ["W", "*"]):
        return f"ACTION: {action}"
    else:
        # If would hit a wall, stay put
        return "ACTION: STAY"


def strategy_flocking_2(obs):
    obs = json.loads(obs)

    # Current position and view
    pos = obs["pos"]
    current_view = obs["views"][-1]

    # Get the center of the view
    view_center = len(current_view) // 2

    # Check valid directions (not walls or out of bounds)
    valid_actions = []

    # Up
    if (view_center - 1 >= 0 and
            current_view[view_center - 1][view_center] not in ["W", "*"]):
        valid_actions.append("UP")

    # Down
    if (view_center + 1 < len(current_view) and
            current_view[view_center + 1][view_center] not in ["W", "*"]):
        valid_actions.append("DOWN")

    # Left
    if (view_center - 1 >= 0 and
            current_view[view_center][view_center - 1] not in ["W", "*"]):
        valid_actions.append("LEFT")

    # Right
    if (view_center + 1 < len(current_view[0]) and
            current_view[view_center][view_center + 1] not in ["W", "*"]):
        valid_actions.append("RIGHT")

    # Add STAY as a valid action
    valid_actions.append("STAY")

    # Choose a random valid action
    action = random.choice(valid_actions)

    return f"ACTION: {action}"


def strategy_flocking_3(obs):
    obs = json.loads(obs)

    # Current position and view
    pos = obs["pos"]
    current_view = obs["views"][-1]
    target = obs["target"]
    messages = obs["messages"]

    # Get agent ID (which is at the center of the view)
    view_center = len(current_view) // 2
    agent_id = int(current_view[view_center][view_center])

    # Get positions where 'A' appears in the target
    target_positions = []
    for i in range(len(target)):
        for j in range(len(target[i])):
            if target[i][j] == 'A':
                target_positions.append([i, j])

    # Check if there's a position assignment in messages
    my_assigned_pos = None
    agent_positions = {}

    for msg in messages:
        try:
            if msg.startswith("POS:"):
                data = json.loads(msg[4:])
                for aid, apos in data.items():
                    agent_positions[int(aid)] = apos
                    if int(aid) == agent_id:
                        my_assigned_pos = apos
        except:
            pass

    # If no assignment yet, create one and broadcast it
    if not my_assigned_pos:
        # Get unassigned positions
        unassigned_positions = [pos for pos in target_positions if pos not in agent_positions.values()]

        if unassigned_positions:
            my_assigned_pos = random.choice(unassigned_positions)
        else:
            # If all positions are taken, choose a random one
            my_assigned_pos = random.choice(target_positions)

        # Update agent_positions with our choice
        agent_positions[agent_id] = my_assigned_pos

        # Create message with our position assignment
        pos_msg = f"POS:{json.dumps(agent_positions)}"
        if len(pos_msg) <= 120:  # Check message length
            message = pos_msg
        else:
            # If too long, just send our own assignment
            message = f"POS:{json.dumps({agent_id: my_assigned_pos})}"
    else:
        # We already have an assignment, so no need for a message
        message = ""

    # Navigate toward assigned position
    di = my_assigned_pos[0] - pos[0]
    dj = my_assigned_pos[1] - pos[1]

    # Choose action based on the largest difference
    action = "STAY"
    if abs(di) > abs(dj):
        if di > 0:
            action = "DOWN"
        elif di < 0:
            action = "UP"
    else:
        if dj > 0:
            action = "RIGHT"
        elif dj < 0:
            action = "LEFT"

    # Check if the chosen action would hit a wall
    new_i, new_j = pos[0], pos[1]
    if action == "UP":
        new_i -= 1
    elif action == "DOWN":
        new_i += 1
    elif action == "LEFT":
        new_j -= 1
    elif action == "RIGHT":
        new_j += 1

    # Get coordinates in the view
    view_i = view_center + (new_i - pos[0])
    view_j = view_center + (new_j - pos[1])

    # Check if the new position is valid
    if (0 <= view_i < len(current_view) and
            0 <= view_j < len(current_view[0]) and
            current_view[view_i][view_j] not in ["W", "*"]):
        # Position is valid, proceed with the action
        pass
    else:
        # If would hit a wall, stay put
        action = "STAY"

    # Return action and optional message
    if message:
        return f"MSG: {message}\nACTION: {action}"
    else:
        return f"ACTION: {action}"


def strategy_synchronization_1(obs):
    import json
    import random

    obs = json.loads(obs)
    current_view = obs["views"][-1]
    light_on = obs["light_on"]
    pos = obs["pos"]
    messages = obs.get("messages", [])

    # Parse messages from other agents
    switch_commands = []
    for msg in messages:
        if msg.startswith("SWITCH:"):
            try:
                command = msg.split(":", 1)[1].strip()
                switch_commands.append(command)
            except:
                pass

    # Find all agents in view and identify leader (lowest ID)
    agents = []
    my_id = None
    for i in range(len(current_view)):
        for j in range(len(current_view[i])):
            cell = current_view[i][j]
            if cell not in ["W", "*", "."]:
                # Extract agent ID and light status
                if cell.startswith("$"):
                    agent_id = int(cell[1:])
                    is_on = False
                else:
                    agent_id = int(cell)
                    is_on = True

                # Check if this is me
                if i == len(current_view) // 2 and j == len(current_view[i]) // 2:
                    my_id = agent_id

                agents.append((agent_id, is_on))

    # Find the leader (agent with lowest ID)
    leader_id = min([agent_id for agent_id, _ in agents]) if agents else None

    # Determine if I'm the leader
    i_am_leader = my_id == leader_id

    # Leader's decision logic
    action = "STAY"
    message = ""

    if i_am_leader:
        # Count agents with lights on and off
        lights_on = sum(1 for _, is_on in agents if is_on)
        lights_off = len(agents) - lights_on

        # Decide whether to switch
        if lights_on == len(agents) or lights_off == len(agents):
            # All agents have the same light state, send switch command
            message = f"SWITCH:{'ON' if not light_on else 'OFF'}"
            action = "SWITCH"
        else:
            # Not all agents have the same state yet
            if light_on and lights_on <= lights_off:
                # More lights are off, so turn mine off too
                action = "SWITCH"
            elif not light_on and lights_off <= lights_on:
                # More lights are on, so turn mine on too
                action = "SWITCH"
    else:
        # Follower logic: obey leader's switch commands
        if switch_commands:
            latest_command = switch_commands[-1]
            if latest_command == "ON" and not light_on:
                action = "SWITCH"
            elif latest_command == "OFF" and light_on:
                action = "SWITCH"

    # Prepare return value with message if needed
    if message:
        return f"MSG: {message}\nACTION: {action}"
    else:
        return f"ACTION: {action}"


def strategy_synchronization_2(obs):
    import json

    obs = json.loads(obs)
    light_on = obs["light_on"]
    history = obs.get("history", [])

    # Get current round number
    current_round = history[-1]["round"] + 1 if history else 0

    # Switch lights based on round number - switch every 10 rounds
    action = "STAY"

    # If round is divisible by 10, switch light state
    if current_round % 10 == 0:
        action = "SWITCH"

    return f"ACTION: {action}"


def strategy_synchronization_3(obs):
    import json

    obs = json.loads(obs)
    current_view = obs["views"][-1]
    light_on = obs["light_on"]
    messages = obs.get("messages", [])

    # Parse incoming messages about light states
    on_count_from_msgs = 0
    off_count_from_msgs = 0
    total_agents_from_msgs = 0

    for msg in messages:
        if msg.startswith("LIGHTS:"):
            try:
                parts = msg.split(":")
                if len(parts) == 4:
                    _, on_count, off_count, total = parts
                    on_count_from_msgs += int(on_count)
                    off_count_from_msgs += int(off_count)
                    total_agents_from_msgs += int(total)
            except:
                pass

    # Count agents with lights on and off in current view
    lights_on = 0
    lights_off = 0
    total_agents = 0

    for row in current_view:
        for cell in row:
            if cell not in ["W", "*", "."]:
                total_agents += 1
                if cell.startswith("$"):
                    lights_off += 1
                else:
                    lights_on += 1

    # Combine view counts with message counts, avoiding double counting
    # by using the message counts as a proportion of what's reported
    if total_agents_from_msgs > 0:
        # Get proportion of lights on/off from messages
        msg_on_ratio = on_count_from_msgs / total_agents_from_msgs
        msg_off_ratio = off_count_from_msgs / total_agents_from_msgs

        # Estimate global state by combining local view with message reports
        estimated_on = lights_on + (msg_on_ratio * total_agents_from_msgs)
        estimated_off = lights_off + (msg_off_ratio * total_agents_from_msgs)
    else:
        estimated_on = lights_on
        estimated_off = lights_off

    # Decide action based on majority state
    action = "STAY"
    if estimated_on > estimated_off and not light_on:
        # Majority is on, but I'm off
        action = "SWITCH"
    elif estimated_off > estimated_on and light_on:
        # Majority is off, but I'm on
        action = "SWITCH"

    # Prepare message to share light state information
    message = f"LIGHTS:{lights_on}:{lights_off}:{total_agents}"

    return f"MSG: {message}\nACTION: {action}"


def strategy_pursuit_1(obs):
    obs = json.loads(obs)

    # Get current position and view
    current_pos = obs["pos"]
    current_view = obs["views"][-1]  # Get the latest view

    # Find prey position in the current view
    prey_pos = None
    for i in range(len(current_view)):
        for j in range(len(current_view[i])):
            if current_view[i][j] == "P":
                prey_pos = [i, j]
                break
        if prey_pos:
            break

    # Calculate center of view
    center_i = len(current_view) // 2
    center_j = len(current_view[0]) // 2

    # Decide action based on prey position
    if prey_pos:
        # Move toward prey
        if prey_pos[0] < center_i:
            return "ACTION: UP"
        elif prey_pos[0] > center_i:
            return "ACTION: DOWN"
        elif prey_pos[1] < center_j:
            return "ACTION: LEFT"
        elif prey_pos[1] > center_j:
            return "ACTION: RIGHT"
        else:
            # Prey is on same position (shouldn't happen)
            return "ACTION: STAY"
    else:
        # No prey visible, move randomly
        actions = ["UP", "DOWN", "LEFT", "RIGHT", "STAY"]

        # Check if we're near walls and avoid them
        if center_i > 0 and current_view[center_i - 1][center_j] == "W":
            actions.remove("UP") if "UP" in actions else None
        if center_i < len(current_view) - 1 and current_view[center_i + 1][center_j] == "W":
            actions.remove("DOWN") if "DOWN" in actions else None
        if center_j > 0 and current_view[center_i][center_j - 1] == "W":
            actions.remove("LEFT") if "LEFT" in actions else None
        if center_j < len(current_view[0]) - 1 and current_view[center_i][center_j + 1] == "W":
            actions.remove("RIGHT") if "RIGHT" in actions else None

        if not actions:
            return "ACTION: STAY"

        return f"ACTION: {random.choice(actions)}"


def strategy_pursuit_2(obs):
    obs = json.loads(obs)

    # Get current position and view
    current_view = obs["views"][-1]  # Get the latest view

    # Calculate center of view
    center_i = len(current_view) // 2
    center_j = len(current_view[0]) // 2

    # Initialize potential field
    potential = {
        "UP": 0,
        "DOWN": 0,
        "LEFT": 0,
        "RIGHT": 0,
        "STAY": -0.5  # Slight preference against staying still
    }

    # Check valid moves
    if center_i > 0 and current_view[center_i - 1][center_j] == "W":
        potential["UP"] = float('-inf')  # Cannot move into walls
    if center_i < len(current_view) - 1 and current_view[center_i + 1][center_j] == "W":
        potential["DOWN"] = float('-inf')
    if center_j > 0 and current_view[center_i][center_j - 1] == "W":
        potential["LEFT"] = float('-inf')
    if center_j < len(current_view[0]) - 1 and current_view[center_i][center_j + 1] == "W":
        potential["RIGHT"] = float('-inf')

    # Add attraction to prey
    prey_seen = False
    for i in range(len(current_view)):
        for j in range(len(current_view[i])):
            cell = current_view[i][j]
            if cell == "P":
                prey_seen = True
                # Calculate direction to prey
                di = i - center_i
                dj = j - center_j

                # Add attraction to the potential field
                if di < 0:
                    potential["UP"] += 10.0 / (abs(di) + 0.1)
                if di > 0:
                    potential["DOWN"] += 10.0 / (abs(di) + 0.1)
                if dj < 0:
                    potential["LEFT"] += 10.0 / (abs(dj) + 0.1)
                if dj > 0:
                    potential["RIGHT"] += 10.0 / (abs(dj) + 0.1)

            # Add repulsion from other agents
            elif cell not in [".", "W", "*"]:
                di = i - center_i
                dj = j - center_j

                # Avoid being too close to other agents
                distance = math.sqrt(di ** 2 + dj ** 2)
                if distance < 2:
                    if di < 0:
                        potential["UP"] -= 2.0 / (distance + 0.1)
                    if di > 0:
                        potential["DOWN"] -= 2.0 / (distance + 0.1)
                    if dj < 0:
                        potential["LEFT"] -= 2.0 / (distance + 0.1)
                    if dj > 0:
                        potential["RIGHT"] -= 2.0 / (distance + 0.1)

    # If no prey is seen, explore randomly
    if not prey_seen:
        valid_actions = [action for action, pot in potential.items() if pot > float('-inf')]
        if valid_actions:
            return f"ACTION: {random.choice(valid_actions)}"
        else:
            return "ACTION: STAY"

    # Choose the action with the highest potential
    best_action = max(potential.items(), key=lambda x: x[1])
    if best_action[1] == float('-inf'):
        return "ACTION: STAY"

    return f"ACTION: {best_action[0]}"


def strategy_pursuit_3(obs):
    obs = json.loads(obs)

    # Get current position and view
    current_pos = obs["pos"]
    current_view = obs["views"][-1]

    # Calculate center of view
    center_i = len(current_view) // 2
    center_j = len(current_view[0]) // 2

    # Find prey position in current view
    prey_pos = None
    for i in range(len(current_view)):
        for j in range(len(current_view[i])):
            if current_view[i][j] == "P":
                prey_pos = [i, j]
                break
        if prey_pos:
            break

    # Parse messages from other agents
    reported_prey_positions = []
    my_id = None

    # Find my ID in the view
    for i in range(len(current_view)):
        for j in range(len(current_view[i])):
            if current_view[i][j] not in [".", "W", "*", "P"]:
                if i == center_i and j == center_j:
                    my_id = current_view[i][j]

    for msg in obs.get("messages", []):
        if msg.startswith("PREY:"):
            try:
                parts = msg.split(":")
                if len(parts) >= 3:
                    # Format: PREY:x,y:agent_id
                    coords = parts[1].split(",")
                    x, y = int(coords[0]), int(coords[1])
                    agent_id = parts[2]

                    # Convert to global coordinates
                    global_x = current_pos[0] - center_i + x
                    global_y = current_pos[1] - center_j + y

                    reported_prey_positions.append({
                        "pos": [global_x, global_y],
                        "agent_id": agent_id
                    })
            except:
                pass

    # Decide message to send
    message = ""
    if prey_pos and my_id:
        # Format: PREY:x,y:agent_id
        message = f"PREY:{prey_pos[0]},{prey_pos[1]}:{my_id}"

    # Decide action
    action = "STAY"

    if prey_pos:
        # We can see the prey directly

        # Check if we're adjacent to the prey
        adjacent_to_prey = (
                (abs(prey_pos[0] - center_i) == 1 and prey_pos[1] == center_j) or
                (abs(prey_pos[1] - center_j) == 1 and prey_pos[0] == center_i)
        )

        if adjacent_to_prey:
            # Stay adjacent to prey, don't move
            action = "STAY"
        else:
            # Calculate direction to prey with slight randomization based on agent ID
            # This helps agents approach from different directions
            di = prey_pos[0] - center_i
            dj = prey_pos[1] - center_j

            # Convert ID to numeric if possible for deterministic behavior
            id_value = 0
            if my_id and my_id.isdigit():
                id_value = int(my_id)

            # Choose direction with some randomization based on ID
            if abs(di) > abs(dj):
                if di < 0:
                    action = "UP"
                else:
                    action = "DOWN"
            elif abs(di) < abs(dj):
                if dj < 0:
                    action = "LEFT"
                else:
                    action = "RIGHT"
            else:
                # Equal distance, use ID to break tie
                if id_value % 2 == 0:
                    if di < 0:
                        action = "UP"
                    else:
                        action = "DOWN"
                else:
                    if dj < 0:
                        action = "LEFT"
                    else:
                        action = "RIGHT"
    elif reported_prey_positions:
        # Use reported positions and move toward prey
        # Sort by ID to ensure different agents go to different sides
        closest = min(reported_prey_positions,
                      key=lambda p: abs(p["pos"][0] - current_pos[0]) + abs(p["pos"][1] - current_pos[1]))

        # Calculate direction
        di = closest["pos"][0] - current_pos[0]
        dj = closest["pos"][1] - current_pos[1]

        # Choose direction with some variation based on agent ID
        if abs(di) > abs(dj):
            if di < 0:
                action = "UP"
            else:
                action = "DOWN"
        else:
            if dj < 0:
                action = "LEFT"
            else:
                action = "RIGHT"
    else:
        # No prey visible, explore randomly
        actions = ["UP", "DOWN", "LEFT", "RIGHT"]

        # Check if we're near walls and avoid them
        if center_i > 0 and current_view[center_i - 1][center_j] == "W":
            actions.remove("UP") if "UP" in actions else None
        if center_i < len(current_view) - 1 and current_view[center_i + 1][center_j] == "W":
            actions.remove("DOWN") if "DOWN" in actions else None
        if center_j > 0 and current_view[center_i][center_j - 1] == "W":
            actions.remove("LEFT") if "LEFT" in actions else None
        if center_j < len(current_view[0]) - 1 and current_view[center_i][center_j + 1] == "W":
            actions.remove("RIGHT") if "RIGHT" in actions else None

        if actions:
            action = random.choice(actions)
        else:
            action = "STAY"

    if message:
        return f"MSG: {message}\nACTION: {action}"
    else:
        return f"ACTION: {action}"


strategies = {
    'Transport1': strategy_transport_1,
    'Transport2': strategy_transport_2,
    'Transport3': strategy_transport_3,
    'Flocking1': strategy_flocking_1,
    'Flocking2': strategy_flocking_2,
    'Flocking3': strategy_flocking_3,
    'Synchronization1': strategy_synchronization_1,
    'Synchronization2': strategy_synchronization_2,
    'Synchronization3': strategy_synchronization_3,
    'Pursuit1': strategy_pursuit_1,
    'Pursuit2': strategy_pursuit_2,
    'Pursuit3': strategy_pursuit_3,
    'Foraging1': strategy_foraging_1,
    'Foraging2': strategy_foraging_2,
    'Foraging3': strategy_foraging_3
}
