from envs.arm_util import RobotArm
import matplotlib.pyplot as plt
    
class Game():
    def __init__(self, setting) -> None:
        self.robotarm = RobotArm(setting['L1'], setting['L2'], setting['obstacles'])
        self.goal = setting['goal']
        self.goal_state = f"move the gripper to {self.goal}, while avoid collision with the obstacles at"
        for x, y, r in setting['obstacles']:
            self.goal_state += f", [{x}, {y}] radius {r}"
        
        self.steps = 0
        self.finish = False
        self.failed = False
        self.shift = setting['shift']
        self.invalid_action = 0
    
    @staticmethod
    def help():
        return """\
1) Move x y: Move the gripper to (x, y).
2) Check: Check robot arm joint positions
3) Help: View the available action options.
"""

    def domain_desc(self)->str:
        return f"""\
The tabletop environment has a robot arm, several obstacles and a goal location.
The robot arm has two joints and a gripper, the goal is to  {self.goal_state} .

The robot arm has the following primitive actions:
1) Move x y:  Move the gripper to (x, y). The rotation of the joints will be calculated by inverse kinematics.
2) Check: Check Robot arm joint positions

The current environment state is:
{self.robotarm.print_joint_positions()}"""


    def start(self) -> str:
        return self.domain_desc()

    def get_goal(self) -> str:
        return self.goal_state

    def task_check(self):
        joint0_pos, joint1_pos, gripper_pos = self.robotarm.get_joint_positions()
        if gripper_pos == self.goal:
            self.finish = True
        plt.close('all')
        return

    def step(self, action: str) -> tuple[bool, str, int]:
        try:
            action = action.lower()
            parts = action.split()
            type = parts[0]
            if type == "move":
                x = float(parts[1])
                y = float(parts[2])
                if self.shift:
                    x -= 0.1
                    y -= 0.1
                result = self.robotarm.move_to(x, y)
            elif type == "check":
                result = self.robotarm.print_joint_positions()
            elif type == "help":
                result = self.help()
                self.steps -= 1
            else:
                result = "Invalid action. Please try again."
                self.invalid_action += 1
                self.steps -= 1
        except Exception as e:
            result = f"Error! {e}"
            self.invalid_action += 1
            self.steps -= 1
        self.steps += 1
        self.task_check()
        return self.finish, result, self.steps