import string
from typing import Dict, List, Tuple

class Object():
    def __init__(self, name:str, pos:Tuple[int, int]) -> None:
        self.name = name
        self.pos = pos

class Robot():
    def __init__(self, pos:Tuple[int, int], inerita=False) -> None:
        self.pos = pos
        self.has_ball = False
        self.last_action = None
        self.inerita = inerita

    def forward(self):
        x, y = self.pos
        y += 1
        if self.inerita and self.last_action == 'b':
            y -= 1
        self.pos = (x, y)
        self.last_action = 'f'

    def backward(self):
        x, y = self.pos
        y -= 1
        if self.inerita and self.last_action == 'f':
            y += 1
        self.pos = (x, y)
        self.last_action = 'b'

    def left(self):
        x, y = self.pos
        x -= 1
        if self.inerita and self.last_action == 'r':
            x += 1
        self.pos = (x, y)
        self.last_action = 'l'

    def right(self):
        x, y = self.pos
        x += 1
        if self.inerita and self.last_action == 'l':
            x -= 1
        self.pos = (x, y)
        self.last_action = 'r'


class Game():
    def __init__(self, setting) -> None:
        self.robot = Robot((0,0), setting['inerita'])
        self.goal = Object("goal", tuple(setting['goal']))
        self.ball = Object("ball", tuple(setting['ball']))

        self.lr_swap = setting['lr_swap']
        self.fb_swap = setting['fb_swap']
        
        self.steps = 0
        self.finish = False
        self.failed = False
        self.invalid_action = 0

    @staticmethod
    def help():
        return """\
1) Forward: Move forward.
2) Backward: Move backward.
3) Left: Move left.
4) Right: Move right.
5) Pos:  Check the current position of the robot.
6) Pick: Pick the ball if robot is at ball location.
7) Reset: Reset the robot to (0, 0) and the ball to original location.
8) Help: View the available action options.
"""

    def robot_pos(self) -> str:
        return f"The robot is located at {self.robot.pos}."

    def domain_desc(self) -> str:
        return f"""\
The environment has a robot, a ball and a goal location.
The goal is to control the robot to pick up the ball, then carry it to goal location.
The ball is located at {self.ball.pos}. The goal is located at {self.goal.pos}.

The robot has the following primitive actions:
Actions:
1) Forward: Move forward 1 unit.
2) Backward: Move backward 1 unit.
3) Left: Move left 1 unit.
4) Right: Move right 1 unit.
5) Pos:  Check the current position of the robot.
6) Pick: Pick the ball if robot is at ball location.
7) Reset: Reset the robot to (0, 0) and drop the ball back to original location.

The current environment state is:
{self.robot_pos()}"""

    def start(self) -> str:
        return self.domain_desc()

    def get_goal(self) -> str:
        return f"The ball is located at {self.ball.pos}. The goal is located at {self.goal.pos}."

    def task_check(self):
        if self.robot.has_ball and self.robot.pos == self.goal.pos:
            self.finish = True

    @staticmethod
    def remove_punctuation(input_string: str) -> str:
        # Create a translation table that maps each punctuation character to None
        translation_table = str.maketrans('', '', string.punctuation)
        # Return the string with punctuation removed
        return input_string.translate(translation_table)

    def step(self, action: str) -> tuple[bool, str, int]:
        try:
            action = action.lower()
            action = self.remove_punctuation(str(action))
            parts = action.split()
            type = parts[0]
            if type == "forward":
                if self.fb_swap:
                    self.robot.backward()
                else:
                    self.robot.forward()
                result = "Robot move forward."
                #result = self.robot_pos()
            elif type == "backward":
                if self.fb_swap:
                    self.robot.forward()
                else:
                    self.robot.backward()
                result = "Robot move backward."
                #result = self.robot_pos()
            elif type == "left":
                if self.lr_swap:
                    self.robot.right()
                else:
                    self.robot.left()
                result = "Robot move left."
                #result = self.robot_pos()
            elif type == "right":
                if self.lr_swap:
                    self.robot.left()
                else:
                    self.robot.right()
                result = "Robot move right."
                #result = self.robot_pos()
            elif type == "pos":
                result = self.robot_pos()
            elif type == "pick":
                if self.robot.pos == self.ball.pos:
                    self.robot.has_ball = True
                if self.robot.has_ball:
                    result = "The robot has grabbed the ball."
                else:
                    result = "The robot is not at ball location."
            elif type == "reset":
                self.robot.pos = (0,0)
                self.robot.has_ball = False
                self.robot.last_action = None
                result = f"Reset robot to (0,0) and ball to {self.ball.pos}."
            elif type == "help":
                result = self.help()
                self.steps -= 1
            else:
                result = "Invalid action. Please try again."
                self.invalid_action += 1
                self.steps -= 1
        except:
            result = "Invalid action. Please try again."
            self.invalid_action += 1
            self.steps -= 1
        self.steps += 1
        self.task_check()
        return self.finish, result, self.steps