import string
from typing import Dict

class Stack():
    def __init__(self, label:str, initial_blocks:list[str], swap=False) -> None:
        self.name = label
        self.swap = swap
        self.initial_blocks = initial_blocks
        self.stack = self.initial_blocks.copy()
    
    def reset(self):
        self.stack = self.initial_blocks.copy()

    def check(self):
        return self.stack

    def size(self):
        return len(self.stack)

    def is_empty(self):
        return len(self.stack) == 0

    def push(self, data):
        self.stack.append(data)

        if self.swap and self.size() > 1:
            self.stack[-1], self.stack[-2] = self.stack[-2], self.stack[-1]

    def pop(self):
        return self.stack.pop()

class Game():
    def __init__(self, setting) -> None:
        self.stacks = {}
        self.repel = None
        if 'repel' in setting:
            self.repel = setting['repel']
        for stack in setting['stacks']:
            self.stacks[stack['label']] = Stack(stack['label'], stack['blocks'], setting['swap'])
        self.init_inventory = setting['inventory']
        self.inventory = self.init_inventory
        
        self.steps = 0
        self.task = {}
        for stack in setting['task']:
            self.task[stack['label']] = Stack(stack['label'], stack['blocks'])
        self.finish = False
        self.failed = False
        self.invalid_action = 0

    @staticmethod
    def help():
        return """\
1) Check: View the blocks currently in all the stack.
2) Pick <stack>: Pick up the top block from the stack and add it to your inventory.
3) Place <stack>: Place the block from your inventory on top of the stack.
4) Inventory: View the blocks currently in the inventory.
5) Reset: Reset all stacks and inventory to initial state.
6) Help: View the available action options.
"""

    def check_stacks(self, stacks:Dict[str, Stack]):
        elements = []
        for stack in stacks.values():
            elements.append(f"Stack {stack.name.upper()} form top to bottom: {stack.check()[::-1]}.")
        return "\n".join(elements)

    def domain_desc(self) -> str:
        return f"""\
The tabletop environment has a robot arm, several colored blocks and stacks.
The robot's inventory can hold only one block at a time. The robot can not pick up any blocks if the inventory is full.
The goal is to arrange the blocks into stacks that match the following goal state:
{self.check_stacks(self.task)}

Initial State:
{self.check_stacks(self.stacks)}

The robot has the following primitive actions:
1) Check: View the blocks currently in all the stack.
2) Pick <stack>: Pick up the top block from the stack and add it to your inventory.
3) Place <stack>: Place the block from your inventory on top of the stack.
4) Inventory: View the blocks currently in the inventory.
5) Reset: Reset all stacks and inventory to initial state.

Here’s how each action works:
- <stack> refers to the label of the stack, such as "A", "B" or any other label for the stack.

For example:
"Pick A" will pick up the top block from stack A and add it to your inventory.

The current environment state is:
{self.check_stacks(self.stacks).replace('\n', " ")}"""

    def start(self) -> str:
        return self.domain_desc()

    def get_goal(self) -> str:
        return self.check_stacks(self.task)

    def check(self) -> str:
        return self.check_stacks(self.stacks).replace('\n', " ")

    def pick(self, label: str) -> str:
        if label in self.stacks:
            stack = self.stacks[label]
            if stack.is_empty():
                return f"Stack {label.upper()} is empty."
            if self.inventory != None:
                return "Inventory is full, cannot pick up new blocks."
            element = stack.pop()
            self.inventory = element
            return f"Pick up the top block from stack {label.upper()} and add it to your inventory."
        else:
            return f"Stack {label} does not exist."

    def place(self, label: str) -> str:
        if label in self.stacks:
            stack = self.stacks[label]
            if self.inventory == None:
                return "Inventory is empty, no block to place."
            if self.repel:
                if self.inventory in self.repel:
                    target = stack.check()
                    if target and target[-1] in self.repel:
                        return f"Failed to place on top of stack {label.upper()}."
            stack.push(self.inventory)
            self.inventory = None
            return f"Place the block from your inventory on top of stack {label.upper()}."
        else:
            return f"Stack {label} does not exist."
    
    def check_inventory(self) -> str:
        if self.inventory:
            return f"The inventory contains a {self.inventory} block."
        else:
            return "The inventory is empty."

    def reset(self) -> str:
        self.inventory = self.init_inventory
        for stack in self.stacks.values():
            stack.reset()
        return "Success! Reset all stacks to their original state."

    def task_check(self):
        for label in self.task:
            if not self.task[label].check() == self.stacks[label].check():
                return
        self.finish = True
        return

    @staticmethod
    def remove_punctuation(input_string: str) -> str:
        # Create a translation table that maps each punctuation character to None
        translation_table = str.maketrans('', '', string.punctuation)
        # Return the string with punctuation removed
        return input_string.translate(translation_table)

    def step(self, action: str) -> tuple[bool, str, int]:
        try:
            action = action.lower().replace('stack', '')
            action = self.remove_punctuation(str(action))
            parts = action.split()
            type = parts[0]
            if type == "pick":
                stack = parts[1]
                result = self.pick(stack)
            elif type == "place":
                stack = parts[1]
                result = self.place(stack)
            elif type == "check":
                result = self.check()
            elif type == "inventory":
                result = self.check_inventory()
            elif type == "reset":
                result = self.reset()
            elif type == "help":
                result = self.help()
                self.steps -= 1
            else:
                result = "Invalid action. Please try again."
                self.invalid_action += 1
                self.steps -= 1
        except:
            result = "Invalid action. Please try again."
            self.invalid_action += 1
            self.steps -= 1
        self.steps += 1
        self.task_check()
        return self.finish, result, self.steps