import os
import json
import envs.mix_color
import envs.robot_arm
import envs.robot_navigation
import envs.block_stacking

# Registry of available environments
ENV = {
    "robot_navigation": envs.robot_navigation.Game,
    "robot_arm": envs.robot_arm.Game,
    "mix_color": envs.mix_color.Game,
    "block_stacking": envs.block_stacking.Game
}

def extract_type(s):
    parts = s.split('_')
    if len(parts) >= 2:
        return '_'.join(parts[:2])
    return s

def load_setting(file_path) -> dict:
    if not os.path.exists(file_path):
        raise FileNotFoundError(f"File not found: {file_path}")
    
    with open(file_path, 'r', encoding='utf-8') as f:
        try:
            return json.load(f)
        except json.JSONDecodeError as e:
            raise ValueError(f"Invalid JSON: {e}")

class EnvManager:
    def __init__(self, task, task_case):
        self.task_case = task_case
        assert 1 <= self.task_case <= 50, "Task case must be between 1 and 50"
        setting_path = f"./tasks/{task}/{task_case}.json"
        self.env_type = extract_type(task)
        assert self.env_type in ENV, f"Environment '{self.env_type}' not registered. Available: {list(ENV.keys())}"
        self.env = ENV[self.env_type](load_setting(setting_path))  # Instantiate the environment

    def start(self) -> str:
        """
        Start the environment and return the task description.
        """
        return self.env.start()

    def step(self, action: str) -> tuple[bool, str, int]:
        """
        Perform one step in the environment using the given action.

        Args:
            action (str): The action to take in the environment.

        Returns:
            tuple:
                - complete (bool): Whether the task is complete.
                - result (str): The result of action.
                - steps (int): The total action steps used in this environment.
        """
        return self.env.step(action)

    def get_desc(self):
        return self.env.domain_desc()

    def get_goal(self):
        return self.env.get_goal()