import mixbox
import webcolors
import string

class Tube():
    def __init__(self, label: str, rgb: tuple[int, int, int], volume: int) -> None:
        self.label = label
        self.rgb = rgb
        self.volume = volume

    def use(self):
        if self.volume>0:
            self.volume -= 1
            return self.rgb
        else:
            return None

class Container():
    def __init__(self, label: str, rgb=None, volume=0) -> None:
        self.label = label
        self.rgb = rgb
        self.volume = volume

    def add(self, rgb: tuple[int, int, int]):
        if self.volume == 0:
            self.rgb = rgb
        else:
            t = 1/(self.volume+1)
            self.rgb = mixbox.lerp(self.rgb,rgb,t)
        self.volume += 1

    def clean(self):
        self.volume = 0
        self.rgb = None

class Game():
    def __init__(self, setting) -> None:
        self.tubes = {}
        for tube in setting['tubes']:
            self.tubes[tube['label']] = Tube(tube['label'], tube['rgb'], tube['volume'])
        self.containers = {}
        for container in setting['containers']:
            self.containers[container['label']] = Container(container['label'], container['rgb'], container['volume'])
        self.steps = 0
        self.task = setting['task']
        self.finish = False
        self.failed = False
        self.invalid_action = 0

    @staticmethod
    def rgb_to_str(rgb) -> str:
        try:
           hex_value = webcolors.rgb_to_hex(rgb)
           return webcolors.hex_to_name(hex_value)
        except ValueError:
            # If exact match not found, find the closest color
            min_colours = {}
            for name in webcolors.names("css3"):
                r_c, g_c, b_c = webcolors.name_to_rgb(name)
                rd = (r_c - rgb[0]) ** 2
                gd = (g_c - rgb[1]) ** 2
                bd = (b_c - rgb[2]) ** 2
                min_colours[(rd + gd + bd)] = name
            return min_colours[min(min_colours.keys())]
    
    @staticmethod
    def help():
        return """\
[Action Options]
1) Check <container> // Check the color of mixture in container.
2) Clean <container> // Clean the mixture in container.
3) Add <tube> to <container> // Add 1 ml of paint from tube into container.
4) Pour <container> into <container> // Pour 1 ml of mixture from a container into another container.
5) Help // Check the action options.
6) Materials // Check available materials.
<tube> refers to the label of the paint tube.
<container> refers to the label of the container.
For example "Add red to A" will add 1 ml of paint from red label tube into container A.
"""

    def materials(self):
        result = ""
        for tube in self.tubes.values():
            result += f"\nTube labeled {tube.label}, with {tube.volume}ml paint."
        for container in self.containers.values():
            result += f"\nContainer labeled {container.label.upper()}."
        return result

    def domain_desc(self):
        return f"""\
The tabletop environment has a robot arm, several paint tubes and containers.
The goal is to: Create {self.task['volume']} ml of {self.task['color']} paint in container {self.task['container'].upper()}.
To complete the task, you must have the required amount of colored paint in the specified container after the robot completes your plan.
You are allowed to use any container to mix the paint.

The robot has the following primitive actions:
1) **Check <container>**: Check the color of mixture in container.
2) **Clean <container>**: Clean the mixture in container.
3) **Add <tube> to <container>**: Add 1 ml of paint from tube into container.
4) **Pour <container> into <container>**: Pour 1 ml of mixture from a container into another container.

Here’s how each action works:
- <tube> refers to the paint tube label, such as "red," "blue," or any other label for the tube.
- <container> refers to the container label, like "A," "B," or any other container.

For example:
1) **Add red to A**: Adds 1 ml of red paint from the red tube to container A.
2) **Check A**: Checks the current color mixture inside container A.

The environment state is:{self.materials()}"""

    def start(self) -> str:
        return self.domain_desc()

    def get_goal(self) -> str:
        return f"Create {self.task['volume']} ml of {self.task['color']} paint in container {self.task['container'].upper()}."

    def check(self, label: str) -> str:
        if label in self.containers:
            container = self.containers[label]
            if container.volume <= 0:
                return f"Container {label.upper()} is empty."
            rgb = container.rgb
            color = self.rgb_to_str(rgb)
            return f"Container {label.upper()} has {container.volume} ml of {color} paint."
        else:
            return f"Container {label} does not exist."

    def clean(self, label: str) -> str:
        if label in self.containers:
            self.containers[label].clean()
            return f"Container {label.upper()} has been cleaned."
        else:
            return f"Container {label} does not exist."

    def add(self, tube, container) -> str:
        if tube not in self.tubes:
            return f"Tube {tube} does not exist."
        if container not in self.containers:
            return f"Container {container} does not exist."
        paste = self.tubes[tube].use()
        if paste is None:
            return f"Tube {tube} is empty."
        self.containers[container].add(paste)
        return f"You add 1 ml of paste from {tube} tube into container {container.upper()}."

    def pour(self, container1, container2) -> str:
        if container1 not in self.containers:
            return f"Container {container1} does not exist."
        if container2 not in self.containers:
            return f"Container {container2} does not exist."
        if self.containers[container1].volume >0:
            self.containers[container1].volume -= 1
            mixture = self.containers[container1].rgb
            self.containers[container2].add(mixture)
            return f"You pour 1 ml of mixure from container {container1.upper()} into container {container2.upper()}."
        else:
            return f"Container {container1.upper()} is empty."

    def task_check(self):
        for tube in self.tubes.values():
            if tube.volume <=0:
                self.failed = True
                
        rgb = self.containers[self.task['container']].rgb
        volume = self.containers[self.task['container']].volume
        if rgb and volume == self.task['volume']:
            color = self.rgb_to_str(rgb)
            if color == self.task['color']:
                self.finish = True

    @staticmethod
    def remove_punctuation(input_string: str) -> str:
        # Create a translation table that maps each punctuation character to None
        translation_table = str.maketrans('', '', string.punctuation)
        # Return the string with punctuation removed
        return input_string.translate(translation_table)

    def step(self, action: str) -> tuple[bool, str, int]:
        try:
            action = self.remove_punctuation(str(action))
            parts = action.lower().split()
            type = parts[0]
            if type == "add" and parts[2] == "to":
                tube = parts[1]
                container = parts[3]
                result = self.add(tube, container)
            elif type == "pour" and parts[2] == "into":
                container1 = parts[1]
                container2 = parts[3]
                result = self.pour(container1, container2)
            elif type == "check":
                container = parts[1]
                result = self.check(container)
            elif type == "clean":
                container = parts[1]
                result = self.clean(container)
            elif type == "help":
                result = self.help()
                self.steps -= 1
            elif type == "materials":
                result = self.materials()
                self.steps -= 1
            else:
                result = "Invalid action. Please try again."
                self.invalid_action += 1
                self.steps -= 1
        except:
            result = "Invalid action. Please try again."
            self.invalid_action += 1
            self.steps -= 1
        self.steps += 1
        self.task_check()
        return self.finish, result, self.steps