import math
import openai
import logging
import requests
import json
import os
import csv
import re
import datetime
import base64
import numpy as np
from collections import Counter
from langchain.chat_models import ChatOpenAI
from langchain.embeddings.openai import OpenAIEmbeddings
from langchain.schema import HumanMessage, SystemMessage
from langchain.vectorstores import Chroma

biome_dictionary = {
    0: "ocean",
    1: "plains",
    2: "desert",
    3: "windswept_hills",
    4: "forest",
    5: "taiga",
    6: "swamp",
    7: "river",
    8: "hell",
    9: "the_end",
    10: "legacy_frozen_ocean",
    11: "frozen_river",
    12: "snowy_plains",
    13: "snowy_mountains",
    14: "mushroom_fields",
    15: "mushroom_field_shore",
    16: "beach",
    17: "desert_hills",
    18: "wooded_hills",
    19: "taiga_hills",
    20: "mountain_edge",
    21: "jungle",
    22: "jungle_hills",
    23: "sparse_jungle",
    24: "deep_ocean",
    25: "stony_shore",
    26: "snowy_beach",
    27: "birch_forest",
    28: "birch_forest_hills",
    29: "dark_forest",
    30: "snowy_taiga",
    31: "snowy_taiga_hills",
    32: "old_growth_pine_taiga",
    33: "giant_tree_taiga_hills",
    34: "windswept_forest",
    35: "savanna",
    36: "savanna_plateau",
    37: "badlands",
    38: "wooded_badlands",
    39: "badlands_plateau",
    40: "warm_ocean",
    41: "deep_warm_ocean",
    42: "lukewarm_ocean",
    43: "deep_lukewarm_ocean",
    44: "cold_ocean",
    45: "deep_cold_ocean",
    46: "frozen_ocean",
    47: "deep_frozen_ocean",
    48: "bamboo_jungle",
    49: "bamboo_jungle_hills",
    129: "sunflower_plains",
    130: "desert_lakes",
    131: "windswept_gravelly_hills",
    132: "flower_forest",
    133: "taiga_mountains",
    134: "swamp_hills",
    149: "modified_jungle",
    151: "modified_jungle_edge",
    155: "old_growth_birch_forest",
    156: "tall_birch_hills",
    157: "dark_forest_hills",
    158: "snowy_taiga_mountains",
    160: "old_growth_spruce_taiga",
    161: "giant_spruce_taiga_hills",
    162: "gravelly_mountains+",
    163: "windswept_savanna",
    164: "shattered_savanna_plateau",
    165: "eroded_badlands",
    166: "modified_wooded_badlands_plateau",
    167: "modified_badlands_plateau",
    178: "soul_sand_valley",
    179: "crimson_forest",
    180: "warped_forest",
    181: "basalt_deltas",
    182: "jagged_peaks",
    183: "frozen_peaks",
    184: "snowy_slopes",
    185: "grove",
    186: "meadow",
    187: "lush_caves",
    188: "dripstone_caves",
    189: "stony_peaks",
    190: "deep_dark",
    191: "mangrove_swamp",
    192: "cherry_grove"
}

def LLM_request(request, max_retries = 5):
    llm = ChatOpenAI( model_name="gpt-4o", temperature=0)
    if max_retries == 0:
        log_info("************Failed to get workflow. Consider updating your prompt.************\n\n")
        return {}
    try:
        llmrequest_system = "Hello! I'm seeking information and advice about Minecraft. I'm hoping you can help answer some questions related to gameplay mechanics, strategies, and item usage."
        llmrequest_query = request
        messages = [
            SystemMessage(content=llmrequest_system),
            HumanMessage(content=llmrequest_query)
        ]

        llm_response = llm(messages)
        llm_response = llm_response.content

        return llm_response
        # debug #######################
    except Exception as e:
        log_info(f"Error arises in WorldModel prediction part: {e} Trying again!\n\n")

        return LLM_request(
            request,
            max_retries=max_retries - 1
        )


def load_json_file(file_path, debug=False):
    try:
        with open(file_path, 'r') as file:
            data = json.load(file)
            if debug:
                print(data)
            return data
    except FileNotFoundError:
        raise FileNotFoundError(f"The file at {file_path} was not found.")
    except json.JSONDecodeError:
        raise ValueError(f"The file at {file_path} contains invalid JSON.")

def load_text(fpaths, by_lines=False):
    with open(fpaths, "r") as fp:
        if by_lines:
            return fp.readlines()
        else:
            return fp.read()

def load_prompt(prompt):
    return load_text(f"/home/**/Workspace/MP5/MP5_agent/agent/prompts/{prompt}.txt")
    # /home/**/Workspace/MP5/MP5_agent/agent/prompts/{prompt}.txt
    # prompts/{prompt}.txt

# Function to encode the image
def encode_image(image_path):
    with open(image_path, "rb") as image_file:
        return base64.b64encode(image_file.read()).decode('utf-8')


def check_biome(biome_id):
    biome = biome_dictionary[biome_id.item()]
    return biome


# TODO gather state infomation after taking each action
def gather_state_info(obs):

    # # Agent State Info.
    
    # obs["equipment"]["name"]
    # obs["equipment"]["quantity"]
    # obs["equipment"]["variant"]
    # obs["equipment"]["cur_durability"]
    # obs["equipment"]["max_durability"]
    names = obs["equipment"]["name"]
    quantities = obs["equipment"]["quantity"]
    equipment_dict = dict(zip(names, quantities))

    # obs["inventory"]["name"]
    # obs["inventory"]["quantity"]
    # obs["inventory"]["variant"]
    # obs["inventory"]["cur_durability"]
    # obs["inventory"]["max_durability"]
    names = obs["inventory"]["name"]
    quantities = obs["inventory"]["quantity"]
    inventory_dict = dict(zip(names, quantities))

    life = obs["life_stats"]["life"]
    oxygen = obs["life_stats"]["oxygen"]
    armor = obs["life_stats"]["armor"]
    food = obs["life_stats"]["food"]
    saturation = obs["life_stats"]["saturation"]
    is_sleeping = obs["life_stats"]["is_sleeping"]
    life_stats_dict = {"life": life, "oxygen": oxygen, "armor": armor, "food": food, "saturation": saturation, "is_sleeping": is_sleeping}

    # # ENV State Info.
    # pos = obs["location_stats"]["pos"]
    biome_id = obs["location_stats"]["biome_id"] # check BOOK
    biome = biome_dictionary[biome_id.item()]
    rainfall = obs["location_stats"]["rainfall"]
    temperature = obs["location_stats"]["temperature"]
    # can_see_sky = obs["location_stats"]["can_see_sky"]
    is_raining = obs["location_stats"]["is_raining"]
    # light_level = obs["location_stats"]["light_level"]
    sky_light_level = obs["location_stats"]["sky_light_level"]
    sun_brightness = obs["location_stats"]["sun_brightness"]
    # location_stats_dict = {"pos": pos, "biome": biome, "rainfall": rainfall, "temperature": temperature, "can_see_sky": can_see_sky, "is_raining": is_raining, "light_level": light_level, "sky_light_level": sky_light_level, "sun_brightness": sun_brightness}
    location_stats_dict = {"biome": biome, "rainfall": rainfall, "temperature": temperature, "is_raining": is_raining, "sky_light_level": sky_light_level, "sun_brightness": sun_brightness}

    state_info = {'equipment':equipment_dict, 'inventory': inventory_dict, 'life_stats': life_stats_dict, 'location_stats': location_stats_dict}
    return state_info


# Custom JSON Encoder for NumPy types including scalar conversions
class NumpyEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, np.ndarray):
            return obj.tolist()  # Convert ndarray to list
        if isinstance(obj, np.integer):
            return int(obj)  # Convert np.int* to int
        if isinstance(obj, np.floating):
            return float(obj)  # Convert np.float* to float
        if isinstance(obj, np.bool_):
            return bool(obj)  # Convert np.bool_ to bool
        return json.JSONEncoder.default(self, obj)

def check_item_availability(names, nums, item, craft_num):
    """
    检查特定物品的数量是否满足要求。

    参数:
    names (list): 物品名称的列表。
    nums (list): 对应物品数量的列表。
    item (str): 需要检查的物品名称。
    craft_num (int): 所需的最少数量。

    返回:
    bool: 如果物品存在且数量大于等于所需数量，则返回True，否则返回False。
    """
    item = item.replace("_", " ")
    for name, num in zip(names, nums):
        if name == item and num >= craft_num:
            return True
    return False

def log_info(info, is_logging=True):
    if not is_logging:
        logging.disable(logging.CRITICAL + 1)

    logging.info(info)

    if not is_logging:
        logging.disable(logging.NOTSET)

    print(info)

def count_inventory(inventory_name_list, inventory_num_list):
    inventory_dict = {}
    for name, num in zip(inventory_name_list, inventory_num_list):
        if name == "air":
            continue

        if name not in inventory_dict:
            inventory_dict[name] = num
        else:
            inventory_dict[name] += num
    return inventory_dict


def share_memory(memory, events):
    inventory_name_list = events['inventory']['name'].tolist()
    inventory_num_list = events['inventory']['quantity'].tolist()
    memory.update_inventory(count_inventory(inventory_name_list, inventory_num_list))

def update_find_obj_name(obj_name):
    if obj_name == "log" or obj_name == "tree":
        return "wood"
    elif obj_name == "cobblestone":
        return "stone"
    elif obj_name =="diamond":
        return "diamond ore"
    else:
        return obj_name


def update_inventory_obj_name(obj_name):
    if obj_name == "wood" or obj_name == "tree":
        return "log"
    elif obj_name == "stone":
        return "cobblestone"
    elif obj_name =="diamond ore":
        return "diamond"
    elif obj_name =="redstone ore" or obj_name =="redstone dust":
        return "redstone"
    elif obj_name =="raw_porkchop" or obj_name =="raw porkchop":
        return "porkchop"
    else:
        return obj_name

# def update_act_plan_obj_names(plan):
#     for step in plan['workflow']:
#         if 'obj' in step['args']:
#             if isinstance(step['args']['obj'], dict):
#                 updated_obj = {}
#                 for key, value in step['args']['obj'].items():
#                     new_key = update_inventory_obj_name(key)
#                     updated_obj[new_key] = value
#                 step['args']['obj'] = updated_obj
#             else:
#                 step['args']['obj'] = update_inventory_obj_name(step['args']['obj'])
#         if 'materials' in step['args']:
#             updated_materials = {}
#             for key, value in step['args']['materials'].items():
#                 new_key = update_inventory_obj_name(key)
#                 updated_materials[new_key] = value
#             step['args']['materials'] = updated_materials
## 更新字典中的所有string，不论是key还是value
def update_dict(data):
    if isinstance(data, dict):
        updated_data = {}
        for key, value in data.items():
            new_key = update_inventory_obj_name(key)
            new_value = update_dict(value)
            updated_data[new_key] = new_value
        return updated_data
    elif isinstance(data, list):
        return [update_dict(item) for item in data]
    elif isinstance(data, str):
        return update_inventory_obj_name(data)
    else:
        return data

def extract_actions(data):
    actions = []
    workflow = data.get("workflow", {})
    action_keys = sorted([key for key in workflow.keys() if key.startswith("action_")], key=lambda x: int(x.split('_')[1]))
    for key in action_keys:
        actions.append(workflow[key])
    return actions

def update_craft_num(craft_name, craft_num):
    if craft_name in ["stick", "planks", "bowl"]:
        craft_num = math.ceil(craft_num / 4) 
    elif craft_name in ["wooden_slab"]:
        craft_num = math.ceil(craft_num / 6) 
    elif craft_name in ["fence", "wooden_door", "iron_door"]:
        craft_num = math.ceil(craft_num / 3) 
    return craft_num


### For generating prompts
def list_dict_to_prompt(list_data):
    if len(list_data) == 0:
        return "- None\n"

    text = ""
    for dict_data in list_data:
        text += dict_to_prompt(dict_data)

    return text

def list_to_prompt(list_data):
    text = ""
    for item in list_data:
        if item:  # Check if item is not None or an empty dictionary/list
            text += f"- {item}\n"
        else:
            text += "- None\n"
    return text


def dict_to_prompt(dict_data):
    text = ""
    for key in dict_data:
        if dict_data[key]:
            text += f"- {key}: {dict_data[key]}\n"
        else:
            text += f"- {key}: None\n"
    return text


def task_to_description_prompt(task_description):
    return f"- description: {task_description}\n" 


def update_find_task_prompt(task_information, find_obj):
    if task_information["description"].lower().find("find") != -1:
        return task_information["description"].capitalize()
    else:
        return f"Find 1 {find_obj}."


def write_task_to_csv(task_id, every_task_max_retries, every_task_max_planning_retries, mode, component, response_metadata):
    # 定义CSV文件的字段
    fields = [
        "task_id",
        "every_task_max_retries",
        "every_task_max_planning_retries",
        "mode",
        "component",
        "completion_tokens",
        "prompt_tokens",
        "total_tokens",
        "model_name",
        "system_fingerprint",
        "finish_reason",
        "logprobs"
    ]

    # 定义CSV文件的路径
    file_path = '/home/**/Workspace/MP5/MP5_agent/agent/tokens_cal/tokens_data.csv'

    # 检查文件是否存在
    file_exists = os.path.exists(file_path)

    # 构建要写入的行数据
    row = {
        "task_id": task_id,
        "every_task_max_retries": every_task_max_retries,
        "every_task_max_planning_retries": every_task_max_planning_retries,
        "mode": mode,
        "component": component,
        "completion_tokens": response_metadata["token_usage"]["completion_tokens"],
        "prompt_tokens": response_metadata["token_usage"]["prompt_tokens"],
        "total_tokens": response_metadata["token_usage"]["total_tokens"],
        "model_name": response_metadata["model_name"],
        # "system_fingerprint": response_metadata["system_fingerprint"],
        # "finish_reason": response_metadata["finish_reason"],
        # "logprobs": response_metadata["logprobs"]
    }

    # 打开文件，追加模式，写入行数据
    with open(file_path, mode='a', newline='') as file:
        writer = csv.DictWriter(file, fieldnames=fields)
        
        # 如果文件不存在，写入表头
        if not file_exists:
            writer.writeheader()
        
        writer.writerow(row)

def write_taskresult_to_csv(planner_search_alg, interval, task_id, task_item, task_level, task_replan, task_success, testModel = 'null'):
    # 定义CSV文件的字段
    fields = [
        "task_id",
        "task_item",
        "task_level",
        "task_replan", 
        "task_success",
        "interval"
    ]

    # 定义CSV文件的路径
    file_path = f'/home/**/Workspace/MP5/MP5_agent/agent/task_result/{testModel}_{planner_search_alg}.csv'\

    # 检查文件是否存在
    file_exists = os.path.exists(file_path)

    # 构建要写入的行数据
    row = {
        "task_id": task_id,
        "task_item": task_item,
        "task_level": task_level,
        "task_replan": task_replan, 
        "task_success": task_success,
        "interval": interval
    }

    # 打开文件，追加模式，写入行数据
    with open(file_path, mode='a', newline='') as file:
        writer = csv.DictWriter(file, fieldnames=fields)
        
        # 如果文件不存在，写入表头
        if not file_exists:
            writer.writeheader()
        
        writer.writerow(row)


def write_rules_using_to_csv(task_id, action_type, rules_list):
    # 定义CSV文件的字段
    fields = [
        "task_id",
        "action_type",
        "rules_list"
    ]

    # 定义CSV文件的路径
    # timestamp = datetime.datetime.now().strftime("%Y%m%d_%H%M%S")
    file_path = f'/home/**/Workspace/MP5/MP5_agent/agent/task_rulesstatistic/rulesstatistic.csv'

    # 检查文件是否存在
    file_exists = os.path.exists(file_path)

    # 构建要写入的行数据
    row = {
        "task_id": task_id,
        "action_type": action_type,
        "rules_list": rules_list
    }

    # 打开文件，追加模式，写入行数据
    with open(file_path, mode='a', newline='') as file:
        writer = csv.DictWriter(file, fieldnames=fields)
        
        # 如果文件不存在，写入表头
        if not file_exists:
            writer.writeheader()
        
        writer.writerow(row)


### For simulating action
def simulate_mine(memory, obj, tool): 
    # Add the mined object to the inventory
    if obj == "wood" or obj =="tree":
        obj = "log"
    elif obj == "stone":
        obj = "cobblestone"
    elif obj == "diamond ore":
        obj = "diamond"

    if obj in memory.inventory:
        memory.inventory[obj] += 1
    else:
        memory.inventory[obj] = 1


def simulate_craft(memory, obj, materials, platform):

    # If all materials are available, craft the object
    for material, quantity in materials.items():
        quantity = int(quantity)
        memory.inventory[material] -= quantity
        if memory.inventory[material] == 0:
            del memory.inventory[material]

    # Add the crafted object to the inventory
    for item, quantity in obj.items():
        quantity = int(quantity)
        if item in memory.inventory:
            memory.inventory[item] += quantity
        else:
            memory.inventory[item] = quantity

def simulate_find(memory, percipient, task_information, file_path="../images/1.jpg", is_del=1):
    while True:
        memory.reset_current_environment_information()
        find_result = percipient.perceive(task_information=task_information, file_path=file_path, is_del=is_del)
        if find_result == 2:
            break