import argparse
import datetime
import os
import os.path as osp

from toolemu.agents import AGENT_TYPES, SIMULATORS
from toolemu.evaluators import EVALUATORS
from toolemu.utils import (
    llm_register_args,
    read_file,
    slice_with_args,
    slicing_register_args,
)

# format YYYY_MM_DD__HH_MM_SS
NOW = datetime.datetime.now().strftime("%Y_%m_%d__%H_%M_%S")

EXECUTE_SCRIPT = "./scripts/emulate.py"
EVALUATE_SCRIPT = "./scripts/evaluate.py"
READRESULTS_SCRIPT = "./scripts/helper/read_eval_results.py"

parser = argparse.ArgumentParser()
llm_register_args(parser, prefix="agent")
slicing_register_args(parser)
parser.add_argument("--dump-dir", "-du", type=str, default="./dumps")
parser.add_argument(
    "--input-case-file", "-inp", type=str, default="./assets/all_cases.json"
)
parser.add_argument(
    "--agent-type",
    "-atp",
    type=str,
    default="naive",
    choices=AGENT_TYPES,
)
parser.add_argument(
    "--simulator-type",
    "-stp",
    type=str,
    default="adv_thought",
    choices=list(SIMULATORS.keys()),
)
parser.add_argument("--batch-size", "-bs", type=int, default=None, help="batch size")
parser.add_argument("--auto", "-auto", action="store_true", help="auto mode")
args = parser.parse_args()


def run_command(cmd, need_confirm=True):
    print("Command:", cmd)
    if not args.auto and need_confirm and input("Continue? (y/n): ") != "y":
        raise RuntimeError("User aborted.")
    if os.system(cmd) != 0:
        raise RuntimeError(f"Command failed: {cmd}")


cases = read_file(args.input_case_file)
cases = slice_with_args(cases, args)
print(f"You are going to run {len(cases)} cases.")
if not args.auto and input("Continue? (y/n): ") != "y":
    exit()

os.makedirs(args.dump_dir, exist_ok=True)
traj_prefix = osp.join(args.dump_dir, "traj")
cmd = (
    f"python {EXECUTE_SCRIPT} -inp {args.input_case_file} -outpre {traj_prefix} -outsuf _{NOW} "
    f"-atp {args.agent_type} -stp {args.simulator_type} -am {args.agent_model_name} -si {args.start_index}"
)
if args.trunc_num is not None:
    cmd += f"-tn {args.trunc_num}"
if args.batch_size is not None:
    cmd += f"-bs {args.batch_size}"
run_command(cmd)
output_path = (
    f"{traj_prefix}_sim_{args.simulator_type}"
    f"_agent_{args.agent_model_name}_{args.agent_type}_{NOW}"
)
print("The trajectories are saved to", output_path)

# evaluate all trajectories for different evaluators
cmd = f"python {EVALUATE_SCRIPT} -inp {output_path}.jsonl"
if args.batch_size is not None:
    cmd += f"-bs {args.batch_size}"
for ev in EVALUATORS.keys():
    run_command(f"{cmd} -ev {ev}")

# read evaluation results
run_command(f"python {READRESULTS_SCRIPT} {output_path}", need_confirm=False)
