import os
import time
import json
import random
import argparse
import numpy as np

import torch
import torch.nn as nn
import torch.utils.data
import torch.distributed as dist

import transformers
from transformers import AutoConfig, AutoTokenizer, AutoModelForCausalLM
from transformers import LlamaForCausalLM as HF_LlamaForCausalLM
from dst_scheduler import DSTScheduler
import datasets
import datasets.distributed
import wandb

from tqdm import tqdm
from loguru import logger
import fnmatch
from peft_pretraining import training_utils, args_utils
from peft_pretraining.dataloader import PreprocessedIterableDataset
from peft_pretraining.modeling_llama import LlamaForCausalLM

# import bitsandbytes as bnb
from galore_torch import GaLoreAdamW, GaLoreAdamW8bit, GaLoreAdafactor

from peft import LoraConfig, get_peft_model
from utils import build_neuro_glia_network, NeuroGliaLayer, onebitlowranklinear
import itertools
import sys
import traceback
from datetime import datetime


class DualWriter:
    def __init__(self, file):
        self.file = file
        self.console = sys.__stdout__

    def write(self, message):
        self.console.write(message)  # Write to the console
        self.file.write(message)     # Write to the file
        self.file.flush()            # Ensure immediate write to file

    def flush(self):
        self.console.flush()
        if not self.file.closed:
            self.file.flush()

# Custom function to log uncaught exceptions globally
def log_uncaught_exceptions(exc_type, exc_value, exc_traceback):
    # Format the exception traceback as a string
    tb_str = "".join(traceback.format_exception(exc_type, exc_value, exc_traceback))

    # Write traceback to both console and log file
    print(f"Uncaught Exception:\n{tb_str}")  # Print stack trace
    logger.error(f"Uncaught Exception:\n{tb_str}")  # Log stack trace with loguru


transformers.logging.set_verbosity_error()

def parse_args(args):
    parser = argparse.ArgumentParser()

    parser.add_argument('--run_name', type=str, required=True)
    parser.add_argument("--model_config", type=str, required=True)
    parser.add_argument("--dataset_name", type=str, required=True)
    parser.add_argument("--use_hf_model", default=False, action="store_true")
    parser.add_argument("--continue_from", type=str, default=None)
    parser.add_argument("--batch_size", type=int, required=True)
    parser.add_argument("--gradient_accumulation", type=int, default=None)
    parser.add_argument("--total_batch_size", type=int, default=None)
    parser.add_argument("--max_length", type=int, default=256)
    parser.add_argument("--optimizer", default="Adam")
    parser.add_argument("--lr", type=float, default=1e-4)
    parser.add_argument("--min_lr_ratio", type=float, default=0.1)
    parser.add_argument("--activation_checkpointing", action="store_true")
    parser.add_argument("--weight_decay", type=float, default=0.0)
    parser.add_argument("--warmup_steps", type=int, default=1_000)
    parser.add_argument("--eval_every", type=int, default=5_000)
    parser.add_argument("--num_training_steps", type=int, default=10_000,
                        help="Number of **update steps** to train for. "
                             "Notice that gradient accumulation is taken into account.")
    parser.add_argument("--max_train_tokens", type=training_utils.max_train_tokens_to_number, default=None,
                        help="Number of tokens to train on. Overwrites num_training_steps. "
                             "You can use M and B suffixes, e.g. 100M or 1B.")
    parser.add_argument("--save_every", type=int, default=10_000)
    parser.add_argument("--save_dir", type=str, default=None)
    parser.add_argument("--tags", type=str, default=None)
    parser.add_argument("--dtype", type=str, default="bfloat16" if torch.cuda.is_bf16_supported() else "float32")
    parser.add_argument("--workers", type=int, default=8)
    parser.add_argument("--seed", type=int, default=0)
    parser.add_argument("--name", type=str, default="test")
    parser.add_argument("--grad_clipping", type=float, default=0.0) 
    parser.add_argument("--scheduler", type=str, default="cosine_restarts", choices=["linear", "cosine", "cosine_restarts"]) 
    # beta1 for adafactor
    parser.add_argument("--beta1", type=float, default=0.0)
    
    # DST parameters
    parser.add_argument("--dst_scheduler", default=False, action="store_true")
    parser.add_argument("--zeta", type=float, default=0.3)
    parser.add_argument("--update_interval", type=int, default=200)
    parser.add_argument("--sparsity", type=float, default=0.99)
    parser.add_argument("--remove_method", type=str, default="weight_magnitude")
    parser.add_argument("--regrow_method", type=str, default="random")
    parser.add_argument("--init_mode", type=str, default="xavier")
    parser.add_argument("--chain_removal", action="store_true")
    parser.add_argument("--T_decay", default="no_decay", type=str)
    parser.add_argument("--adaptive_zeta", action="store_true")
    parser.add_argument("--WS", action="store_true")
    parser.add_argument("--ws_beta", default=0.25, type=float)
    parser.add_argument("--itop", action="store_true", help="activate ITOP analysis")
    parser.add_argument("--EM_S", action="store_true", help="EM_S solution")
    parser.add_argument("--factor", type=float, default=0.01)
    parser.add_argument("--granet", action="store_true", help="granet solution")
    parser.add_argument("--granet_init_sparsity", default=0.9, type=float)
    parser.add_argument("--granet_init_step", default=0, type=int)
    parser.add_argument("--ssam", action="store_true", help="ssam solution")
    parser.add_argument("--history_weights", action="store_true", help="save the weights of the model after each evolutionary epoch")
    parser.add_argument("--new_history_weights", action="store_true", help="save the weights of the model after each evolutionary epoch")
    parser.add_argument("--gmp", action="store_true")
    parser.add_argument("--pruning_scheduler", type=str, default="none", help="none, linear, granet, s_shape")
    parser.add_argument("--pruning_method", type=str, default="none", help="ri, weight_magnitude, MEST")
    parser.add_argument("--sparsity_distribution", type=str, default="uniform", help="uniform, non-uniform")
    parser.add_argument("--early_stop", action="store_true")
    parser.add_argument("--pruning_T_end", default=None, type=float, help="pruning T end")
    parser.add_argument("--save_new", action="store_true")
    parser.add_argument("--saving_steps", default=10, type=float)
    parser.add_argument("--adaptive_shield",  action="store_true")
    parser.add_argument("--shield_threshold", default=0.05, type=float)
    parser.add_argument("--start_T", default=1, type=float)
    parser.add_argument("--end_T", default=1, type=float)
    
    parser.add_argument("--sltrain", action="store_true")
    parser.add_argument("--onebitlowranktrain", action="store_true")
    parser.add_argument("--lora_alpha", type=float, default=32)
    parser.add_argument("--lora_dropout", type=float, default=0.)
    parser.add_argument("--train_scaling", default=False, action="store_true")
    parser.add_argument("--rank", type=int, default=128)

    parser.add_argument("--not_contextual", default=False, action="store_true")
    parser.add_argument("--residual_modulator", default=False, action="store_true")
    parser.add_argument("--functional_modulator", default=False, action="store_true")
    parser.add_argument("--modulator_mode", type=str, choices=["laurel-rw1", "laurel-rw2", "laurel-rw3", "laurel-rw4", "laurel-rw5", "laurel-lr", "laurel-pa", "comt", "comt-balance"], default="comt")
    parser.add_argument("--laurel_k", type=int, default=4)
    parser.add_argument("--laurel_cache_len", type=int, default=8)
    parser.add_argument("--laurel_r", type=int, default=4)
    parser.add_argument("--channel_wise", default=False, action="store_true")
    parser.add_argument("--scalar_wise", default=False, action="store_true")

    
    
    parser.add_argument("--neuro_glia_network", action="store_true")
    parser.add_argument("--hidden_size", type=int, default=2)
    parser.add_argument("--nonlinear_function", type=str, choices=["sigmoid", "softmax", "learnable_sigmoid"], default="sigmoid")
    parser.add_argument("--use_input_cmgllf", action="store_true", help="use cm-gllf as the nonlinear function")
    parser.add_argument("--use_output_cmgllf", action="store_true", help="use cm-gllf as the nonlinear function")

    # disable ddp, single_gpu
    parser.add_argument("--single_gpu", default=False, action="store_true")
    parser.add_argument("--no_log", default=False, action="store_true")
    parser.add_argument("--no_decay", default=False, action="store_true")
    parser.add_argument("--iterative_warmup_steps", type=int, default=0)
    parser.add_argument("--log_to_file", default=False, action="store_true")
    parser.add_argument("--only_save_last", default=False, action="store_true")
    parser.add_argument("--save_path", type=str, default=None)


    
    args = parser.parse_args(args)
    args.run_name=f"step_{args.num_training_steps}_lr_{args.lr}"

    if args.neuro_glia_network:   
        args.run_name+= f"neuroglia_hidden_size_{args.hidden_size}_nonlinear_{args.nonlinear_function}_{args.not_contextual}_residual_modulator_{args.residual_modulator}_functional_modulator_{args.functional_modulator}_not_contextual_{args.not_contextual}"

        if "laurel-rw" in args.modulator_mode:
            args.run_name+= f"_{args.modulator_mode}"

        elif args.modulator_mode == "laurel-lr":
            args.run_name+= f"_laurel-lr_{args.laurel_r}"

        elif args.modulator_mode == "laurel-pa":
            args.run_name+= f"_laurel-pa_k_{args.laurel_k}_cachelen_{args.laurel_cache_len}"

        elif "comt" in args.modulator_mode:
            args.run_name+= f"_{args.modulator_mode}"

        if args.use_input_cmgllf:
            args.run_name+= f"_use_input_cmgllf"
        if args.use_output_cmgllf:
            args.run_name+= f"_use_output_cmgllf"

        if args.channel_wise:
            args.run_name+= f"_channel_wise"

        if args.scalar_wise:
            args.run_name+= f"_scalar_wise"

    print(args.run_name)
    args = args_utils.check_args_torchrun_main(args)
    return args


@torch.no_grad()
def evaluate_model(model, preprocess_batched, pad_idx, global_rank, world_size, device, batch_size, dataset_name, train_valid_data):
    model.eval()
    _time = time.time()
    if args.dataset_name == "openwebtext":
        val_data = train_valid_data['test']
        remove_columns = ['text']
    elif args.dataset_name == "c4":
        val_data = datasets.load_dataset("c4", "en", split="validation", trust_remote_code=True)
        remove_columns = ["text", "timestamp", "url"]
    val_data = val_data.shuffle(seed=42)
    logger.info(f"Loaded validation dataset in {time.time() - _time:.2f} seconds")

    if not args.single_gpu:
        val_data = datasets.distributed.split_dataset_by_node(val_data, rank=global_rank, world_size=world_size)

    val_data_mapped = val_data.map(
        preprocess_batched,
        batched=True,
        remove_columns=remove_columns,
    )
    val_data_mapped.batch = lambda batch_size: training_utils.batch_fn(val_data_mapped, batch_size)

    target_eval_tokens = 10_000_000
    evaluated_on_tokens = 0
    total_loss = torch.tensor(0.0).to(device)
    total_batches = 1
    logger.info(f"Eval set prepared in {time.time() - _time:.2f} seconds")

    for batch in val_data_mapped.batch(batch_size=batch_size):
        if evaluated_on_tokens > target_eval_tokens:
            break
        total_batches += 1

        batch = {k: v.to(device) for k, v in batch.items()}
        labels = batch["input_ids"].clone()
        labels[labels == pad_idx] = -100
        loss = model(**batch, labels=labels).loss
        total_loss += loss.detach()

        evaluated_on_tokens += (batch["input_ids"] != pad_idx).sum().item() * world_size

    total_loss = total_loss / total_batches

    # Gather losses across all GPUs
    gathered_losses = [torch.zeros_like(total_loss) for _ in range(world_size)]
    dist.all_gather(gathered_losses, total_loss)
    total_loss = sum([t.item() for t in gathered_losses]) / world_size

    return total_loss, evaluated_on_tokens


def main(args):
    torch.manual_seed(args.seed)
    np.random.seed(args.seed)
    random.seed(args.seed)

    assert "LOCAL_RANK" in os.environ, "torchrun should set LOCAL_RANK"
    global_rank = int(os.environ['RANK'])
    local_rank = int(os.environ["LOCAL_RANK"])
    world_size = int(os.environ["WORLD_SIZE"])
    torch.cuda.set_device(local_rank)
    logger.info(f"world_size: {world_size}")
    logger.info(f"Global rank {global_rank}, local rank {local_rank}, device: {torch.cuda.current_device()}")

    dist.init_process_group(backend="nccl", rank=global_rank, world_size=world_size)

    logger.info("Process group initialized")
    device = f"cuda:{local_rank}"

    if args.total_batch_size is not None:
        if args.gradient_accumulation is None:
            assert args.total_batch_size % world_size == 0, "total_batch_size must be divisible by world_size"
            args.gradient_accumulation = args.total_batch_size // (args.batch_size * world_size)
            assert args.gradient_accumulation > 0, "gradient_accumulation must be greater than 0"

    assert args.gradient_accumulation * args.batch_size * world_size == args.total_batch_size, \
        "gradient_accumulation * batch_size * world_size must be equal to total_batch_size"

    # turn off logger
    if global_rank != 0: logger.remove()

    model_name = os.path.splitext(os.path.basename(args.model_config))[0]
    # initialize wandb without config (it is passed later)
    if global_rank == 0:
        wandb.init(project=f"neuro-glia-{args.dataset_name}", name=f"{model_name}_{args.run_name}", mode="disabled" if args.no_log else "online")

    if args.log_to_file is True and global_rank == 0:
        now = datetime.now()
        date_time_str = now.strftime("%Y-%m-%d_%H-%M-%S")
        log_file_name = f'tmp/galore-{args.dataset_name}-{model_name}_{args.run_name}_{date_time_str}'
        log_file = open(f"{log_file_name}.txt", 'w')
        sys.stdout = DualWriter(log_file)
        logger.remove()  # Remove default handler
        logger.add(sys.stdout, format="{time} {level} {message}", level="INFO")  # Add logging to console
        logger.add(f"{log_file_name}.log", format="{time} {level} {message}", level="INFO",
                   rotation="10 MB")  # Add logging to file
        sys.excepthook = log_uncaught_exceptions
        
    logger.info(f"Using dist with rank {global_rank} (only rank 0 will log)")
    logger.info("*" * 40)
    logger.info(f"Starting training with the arguments")
    for k, v in vars(args).items():
        logger.info(f"{k:30} {v}")
    logger.info("*" * 40)

    train_valid_data = None

    if args.dataset_name == "openwebtext":
        train_valid_data = datasets.load_dataset('openwebtext', split='train', trust_remote_code=True)
        # train_valid_data = datasets.load_dataset('../../dataset/openwebtext', split='train')

        train_valid_data = train_valid_data.train_test_split(test_size=0.05, seed=args.seed)
        data = train_valid_data["train"]
    elif args.dataset_name == "c4":
        data = datasets.load_dataset("c4", "en", split="train", trust_remote_code=True)

    seed_for_shuffle = 42
    
    logger.info(f"Shuffling data with seed {seed_for_shuffle}")
    data: datasets.Dataset = data.shuffle(seed=seed_for_shuffle)
    if not args.single_gpu:
        data = datasets.distributed.split_dataset_by_node(
            data, rank=global_rank, world_size=world_size,
        )

    # it doesn't matter which tokenizer we use, because we train from scratch
    # T5 tokenizer was trained on C4 and we are also training on C4, so it's a good choice
    while True:
        try:
            tokenizer = AutoTokenizer.from_pretrained("t5-base", model_max_length=args.max_length)
            break
        except:
            logger.warning("Failed to load tokenizer, retrying...")
            time.sleep(1)

    def preprocess_batched(batch):
        batch = tokenizer(
            batch["text"],
            max_length=args.max_length,
            truncation=True,
            padding="max_length",
            return_tensors="pt",
        )
        return batch


    dataset = PreprocessedIterableDataset(data, tokenizer, batch_size=args.batch_size, max_length=args.max_length)
    dataloader = torch.utils.data.DataLoader(dataset, batch_size=None, num_workers=args.workers)
    
    model_config = AutoConfig.from_pretrained(args.model_config)
    model_config.residual_modulator = args.residual_modulator
    model_config.residual_modulator_hidden_size = args.hidden_size
    model_config.modulator_mode = args.modulator_mode
    model_config.laurel_k = args.laurel_k
    model_config.cache_len = args.laurel_cache_len
    model_config.laurel_r = args.laurel_r

    if args.use_hf_model:
        model: HF_LlamaForCausalLM = AutoModelForCausalLM.from_config(model_config)
    else:
        model = LlamaForCausalLM(model_config)


    # if args.activation_checkpointing:
    #     model.gradient_checkpointing_enable()

    global_step = 0
    update_step = 0
    beginning_step = 0
    tokens_seen = 0
    tokens_seen_before = 0

    if args.functional_modulator:
        model = build_neuro_glia_network(model, args)
    
    if args.dtype in ["bf16", "bfloat16"]:
        model = model.to(device=device, dtype=torch.bfloat16)
    else:
        model = model.to(device=device)




    n_total_params = sum(p.numel() for p in model.parameters())
    
    # Initialize wandb
    run_config = dict(vars(args))
    run_config.update({
        "max_lr": run_config.pop("lr"),  # rename lr to max_lr to avoid conflicts with scheduler
        "total_params_M": n_total_params / 1_000_000,
        "dataset": args.dataset_name,
        "model": model_config.to_dict(),
        "world_size": world_size,
        "device": str(device),
        "a_screen_id": os.environ.get('STY', 'Not in a screen session')
    })
    
    if global_rank == 0:
        wandb.config.update(run_config, allow_val_change=True)
        wandb.save(os.path.abspath(__file__), policy="now") # save current script
        # fix tqdm visual length to 80 so that the progress bar
        # doesn't jump around when changing from external display to laptop
        pbar = tqdm(total=args.num_training_steps - update_step, desc="Update steps", ncols=80)
    

    # print params and trainable params
    logger.info(f"\n{model}\n")
    logger.info(f"Total params: {sum(p.numel() for p in model.parameters()) / 1_000_000:.2f}M")
    logger.info(f"Trainable params: {sum(p.numel() for p in model.parameters() if p.requires_grad) / 1_000_000:.2f}M")
    
    if args.residual_modulator and args.modulator_mode in ["laurel-rw1", "laurel-rw2", "laurel-rw3", "laurel-rw4", "laurel-rw5"]:
        # 模型参数分组
        modulator_params = []
        base_params = []

        for name, param in model.named_parameters():
            if any(key in name for key in [
                "attn_residual_modulator_alpha",
                "attn_residual_modulator_beta",
                "ffn_residual_modulator_alpha",
                "ffn_residual_modulator_beta"
            ]):
                modulator_params.append(param)
            else:
                base_params.append(param)

        # if args.optimizer.lower() == "adam":
        #     optimizer = torch.optim.Adam(trainable_params, lr=args.lr, weight_decay=args.weight_decay)
        # # implement sgd
        # elif args.optimizer.lower() == "sgd":
        #     optimizer = torch.optim.SGD(trainable_params, lr=args.lr, weight_decay=args.weight_decay, momentum=args.beta1)


        optimizer = torch.optim.Adam([
            {"params": base_params, "lr": args.lr},
            {"params": modulator_params, "lr": args.lr * 0.1},
        ], weight_decay=args.weight_decay)
    else:
        trainable_params = [p for p in model.parameters() if p.requires_grad]
        if args.optimizer.lower() == "adam":
            optimizer = torch.optim.Adam(trainable_params, lr=args.lr, weight_decay=args.weight_decay)
        # implement sgd
        elif args.optimizer.lower() == "sgd":
            optimizer = torch.optim.SGD(trainable_params, lr=args.lr, weight_decay=args.weight_decay, momentum=args.beta1)
    
        # # 8-bit Adam
        # elif args.optimizer.lower() == "adam8bit":
        #     optimizer = bnb.optim.Adam8bit(trainable_params, lr=args.lr, weight_decay=args.weight_decay)
        
        else:
            raise ValueError(f"Optimizer {args.optimizer} not supported")


    scheduler = training_utils.get_scheculer(
        optimizer=optimizer,
        scheduler_type=args.scheduler,
        num_training_steps=args.num_training_steps,
        warmup_steps=args.warmup_steps,
        min_lr_ratio=args.min_lr_ratio,
        restart_warmup_steps=args.iterative_warmup_steps,
        cycle_length=args.update_interval,
        no_decay=args.no_decay,
    )


    T_end = int(args.num_training_steps*0.75)
    if args.pruning_T_end is None:
        args.pruning_T_end = T_end

    if args.dst_scheduler:
        pruner = DSTScheduler(model, optimizer, alpha=args.zeta, delta=args.update_interval, static_topo=False, T_end=int(args.num_training_steps*0.75), ignore_linear_layers=False, sparsity_distribution=args.sparsity_distribution, grad_accumulation_n=args.gradient_accumulation, args=args)

    

    if args.continue_from is not None:
        logger.info("*" * 40)
        logger.info(f"Loading model from {args.continue_from}")
        checkpoint_path = os.path.join(args.continue_from, "all_model.pt")
        state_dict = torch.load(checkpoint_path, map_location="cpu")
        prefix = "module."
        state_dict = {key.replace(prefix, ""): value for key, value in state_dict.items()}
        model.load_state_dict(state_dict, strict=True)
        logger.info(f"Model successfully loaded (strict=True policy)")
        for n, m in model.named_modules():
            if isinstance(m, LoraLayer):
                print(f"{n}")
                print_lora(m, "reload")

        if os.path.exists(os.path.join(args.continue_from, "training_state.json")):
            logger.info(
                f"Loading training state like global_step, update_step, and tokens_seen from {args.continue_from}")
            with open(os.path.join(args.continue_from, "training_state.json")) as f:
                _old_state = json.load(f)
            global_step = _old_state["global_step"]
            update_step = _old_state["update_step"]
            tokens_seen = _old_state["tokens_seen"]
            tokens_seen_before = _old_state["tokens_seen_before"]
            logger.info(f"global_step       : {global_step}")
            logger.info(f"update_step       : {update_step}")
            logger.info(f"tokens_seen       : {tokens_seen}")
            logger.info(f"tokens_seen_before: {tokens_seen_before}")
            logger.info(f"Will train for {args.num_training_steps - update_step} update steps")
        else:
            logger.warning(f"Did not find training state in {args.continue_from}, global step will start from zero")

        optimizer_checkpoint = torch.load(os.path.join(args.continue_from, "optimizer.pt"), map_location="cpu")
        optimizer.load_state_dict(optimizer_checkpoint["optimizer"])
        pruner.load_state_dict(optimizer_checkpoint["dst_scheduler"])
        # update_step = optimizer_checkpoint["update_step"]
        # global_step = optimizer_checkpoint["global_step"]
        logger.info(f"Optimizer and scheduler restored from {args.continue_from}")
        logger.info("*" * 40)

    if not args.single_gpu:
        model: LlamaForCausalLM = torch.nn.parallel.DistributedDataParallel(
            model,
            device_ids=[local_rank],
            output_device=local_rank,
            broadcast_buffers=False,
        )
    # for n, m in model.named_modules():
    #     if isinstance(m, NeuroGliaLayer):
    #         m.update_mask_device()

    # global steps and others are defined above
    pad_idx = tokenizer.pad_token_id
    update_time = time.time()
    model.module.generation_config.pad_token_id = pad_idx
    local_step = 0  # when continue_from is used, local_step != global_step

    # ##############################
    # TRAINING LOOP
    # we'll never go through all the data, so no need for epochs
    # ##############################


    for batch_idx, batch in enumerate(dataloader):

        global_step += 1
        local_step += 1

        if update_step > args.num_training_steps:
            logger.info(f"Reached max number of update steps (f{args.num_training_steps}). Stopping training.")
            print(f"Rank {global_rank} stopping training.")
            break

        batch = {k: v.to(device) for k, v in batch.items()}
        labels = batch["input_ids"].clone()
        labels[labels == pad_idx] = -100
        tokens_seen += (batch["input_ids"] != pad_idx).sum().item() * world_size

        loss = model(**batch, labels=labels).loss

        # loss_efficiency = 0
        # if args.efficiency_loss:
            
        #     ii = 0
        #     for n,m in model.named_modules():
        #         if isinstance(m, NeuroGliaLayer):
        #             ii += 1
        #             loss_efficiency += ((m.experts_prob*m.experts_prob).mean(dim=-1) * batch['attention_mask']).sum()/ batch['attention_mask'].sum()
        #     loss_efficiency = loss_efficiency / ii
        #     loss += loss_efficiency * args.efficiency_factor

        # loss_separability = 0
        # if args.separability_loss:
            
        #     ii = 0
        #     for n,m in model.named_modules():
        #         if isinstance(m, NeuroGliaLayer):
        #             ii += 1
        #             loss_separability += ((1 / ((m.experts_prob - 0.5) * (m.experts_prob - 0.5) + 1)).mean(dim=-1) * batch['attention_mask']).sum() / batch['attention_mask'].sum()
        #     loss_separability = loss_separability / ii
        #     loss += loss_separability * args.separability_factor
        scaled_loss = loss / args.gradient_accumulation
        scaled_loss.backward()

        if global_step % args.gradient_accumulation != 0:
            continue


        # The below code is only executed during the update step
        
        # add grad clipping
        if args.grad_clipping != 0.0: torch.nn.utils.clip_grad_norm_(trainable_params, args.grad_clipping)

        if global_rank == 0: pbar.update(1)
        
        if args.dst_scheduler:
            if pruner():
                optimizer.step()
        else:
            for l, m in model.named_modules():
                if isinstance(m, onebitlowranklinear):
                    m._update_weight()
            optimizer.step()
        scheduler.step()
        optimizer.zero_grad()


        update_step += 1
        update_time = time.time() - update_time

        # save checkpoint by save_every
        if local_step > args.gradient_accumulation and update_step % args.save_every == 0 and global_rank == 0:
            if args.only_save_last is True:
                current_model_directory = args.save_dir
            else:
                current_model_directory = f"{args.save_dir}/model_{update_step}"
            logger.info(f"Saving model and optimizer to {current_model_directory}, update step {update_step}")
            os.makedirs(args.save_dir, exist_ok=True)

            torch.save(model.state_dict(), os.path.join(current_model_directory, "all_model.pt"))
            # model.module.save_pretrained(current_model_directory, max_shard_size='100GB')

            optimizer_checkpoint = {
                "optimizer": optimizer.state_dict(),
                "update_step": update_step,
                "global_step": global_step,
                "config": run_config,
                "wandb": wandb.run.dir,
                "dtype": args.dtype,
            }
            torch.save(optimizer_checkpoint, f"{current_model_directory}/optimizer.pt")

            training_state_checkpoint = {
                "global_step": global_step,
                "update_step": update_step,
                "tokens_seen": tokens_seen,
                "tokens_seen_before": tokens_seen_before,
                "update_time": update_time,
            }
            with open(f"{current_model_directory}/training_state.json", "w") as f:
                json.dump(training_state_checkpoint, f, indent=4)
                
            # save wandb related info
            wandb_info = {
                "wandb_id": wandb.run.id,
            }
            with open(f"{args.save_dir}/wandb.json", "w") as f:
                json.dump(wandb_info, f, indent=4)

        # evaluation
        if update_step % args.eval_every == 0:
            

            logger.info(f"Performing evaluation at step {update_step}")
            total_loss, evaluated_on_tokens = evaluate_model(
                model, preprocess_batched, pad_idx, global_rank, world_size, device, args.batch_size, args.dataset_name, train_valid_data
            )
            if global_rank == 0:
                wandb.log({
                    "final_eval_loss": total_loss,
                    "final_eval_tokens": evaluated_on_tokens,
                    },
                    step=update_step,
                )

                print(f"update_step: {update_step}", {
                    "final_eval_loss": total_loss,
                    "final_eval_tokens": evaluated_on_tokens,
                    })
            logger.info(f"Eval loss at step {update_step}: {total_loss}")
            model.train()
        lr = optimizer.param_groups[0]["lr"]

        tokens_in_update = tokens_seen - tokens_seen_before
        tokens_seen_before = tokens_seen
        batches_in_update = args.gradient_accumulation * world_size

        if global_rank == 0:
            wandb.log({
                "loss": loss.item(),
                "lr": lr,
                "update_step": update_step,
                "tokens_seen": tokens_seen,
                "throughput_tokens": tokens_in_update / update_time,
                "throughput_examples": args.total_batch_size / update_time,
                "throughput_batches": batches_in_update / update_time,
                },
                step=update_step,
            )
            
            print(f"update_step: {update_step},", {
                "loss": loss.item(),
                "lr": lr,
                "update_step": update_step,
                "tokens_seen": tokens_seen,
                "throughput_tokens": tokens_in_update / update_time,
                "throughput_examples": args.total_batch_size / update_time,
                "throughput_batches": batches_in_update / update_time,
                })
        update_time = time.time()

    # ##############################
    # END of training loop
    # ##############################
    logger.info("Training finished")
    if global_rank == 0: pbar.close()

    if args.only_save_last is True:
        current_model_directory = f"{args.save_dir}/"
    else:
        current_model_directory = f"{args.save_dir}/model_{update_step}"
    if global_rank == 0:
        # logger.info(f"Saving model and optimizer to {current_model_directory}, update step {update_step}")
        # os.makedirs(args.save_dir, exist_ok=True)
        # model.module.save_pretrained(f"trained_model/{args.run_name}")
        tokenizer.save_pretrained(f"trained_model/{args.run_name}")
        torch.save(model.state_dict(), os.path.join(current_model_directory, "all_model.pt"))

        # optimizer_checkpoint = {
        #     "optimizer": optimizer.state_dict(),
        #     "update_step": update_step,
        #     "global_step": global_step,
        #     "config": run_config,
        #     "wandb": wandb.run.dir,
        #     "dtype": args.dtype,
        # }
        # torch.save(optimizer_checkpoint, f"{current_model_directory}/optimizer.pt")

        training_state_checkpoint = {
            "global_step": global_step,
            "update_step": update_step,
            "tokens_seen": tokens_seen,
            "tokens_seen_before": tokens_seen_before,
            "update_time": update_time,
        }
        # with open(f"{current_model_directory}/training_state.json", "w") as f:
        #     json.dump(training_state_checkpoint, f, indent=4)

    # Final evaluation
    logger.info("Running final evaluation")
    model.eval()
    
    del loss, optimizer
    import gc; gc.collect()
    torch.cuda.empty_cache()

    total_loss, evaluated_on_tokens = evaluate_model(
        model, preprocess_batched, pad_idx, global_rank, world_size, device, args.batch_size, args.dataset_name, train_valid_data
    )

    if global_rank == 0:
        wandb.log({
            "final_eval_loss": total_loss,
            "final_eval_tokens": evaluated_on_tokens,
            },
            step=update_step,
        )
        
        print(f"update_step: {update_step},", {
        "final_eval_loss": total_loss,
        "final_eval_tokens": evaluated_on_tokens,
        })
        logger.info(f"Final eval loss: {total_loss}")

    logger.info("Script finished successfully")
    print(f"Rank {global_rank} finished successfully")

    
    if args.log_to_file is True and global_rank == 0:
        sys.stdout = sys.__stdout__

        # Close the file
        log_file.close()


    
if __name__ == "__main__":
    print("Starting script")
    args = parse_args(None)
    main(args)
