#include "../../src/utils/utils.h"
#include "../../src/bloom/bloom.h"
#include "../utils/str2uint64_hash.h"
#include <fstream>
#include <iostream>
#include <string>
#include <vector>
#include <unistd.h>

void construct_bloom(
    const std::vector<std::string>& all_pos_key_str,
    std::string bloom_folder_path,
    float F
) {
    // If bloom_folder_path does not exist, create it
    if (!std::filesystem::exists(bloom_folder_path)) {
        std::filesystem::create_directories(bloom_folder_path);
    }

    std::string bloom_model_path = bloom_folder_path + "/bloom_model.bin";
    std::string bloom_info_path = bloom_folder_path + "/info.json";

    // Construct bloom
    Bloom bloom = Bloom();
    float add_keys_time_ms = 0;
    float model_construct_time_ms = 0;
    {
        auto start = std::chrono::high_resolution_clock::now();

        // prepare positive samples
        std::vector<uint64_t> all_pos_key;
        for (int i = 0; i < (int)all_pos_key_str.size(); i++) {
            all_pos_key.push_back(str2uint64_hash(all_pos_key_str[i]));
        }

        // Configure bloom
        bloom.configure_bloom(all_pos_key.size(), F);

        // add keys
        bloom.add_all(all_pos_key);

        auto end = std::chrono::high_resolution_clock::now();
        model_construct_time_ms = std::chrono::duration_cast<std::chrono::milliseconds>(end - start).count();
        add_keys_time_ms = std::chrono::duration_cast<std::chrono::milliseconds>(end - start).count();

        // Check FNR == 0
        /*
        {
            std::vector<bloomfilter::Status> predictions_ = bloom.contains_all(X_val_key_pos);
            int false_negatives = 0;
            for (int i = 0; i < (int)predictions_.size(); i++) {
                if (predictions_[i] == bloomfilter::Status::NotFound) {
                    false_negatives++;
                }
            }
            if (false_negatives > 0) {
                std::cerr << "Error: False negatives found in the positive samples.\n";
                exit(1);
            }
        }
        */
    }

    // Save bloom
    float model_size_kb = 0;
    {
        bloom.save_model(bloom_model_path);
        model_size_kb = getBinFileSize(bloom_model_path);
    }

    // Save info.json
    {
        std::ofstream ofs(bloom_info_path);
        ofs << "{\n";
        ofs << "  \"model_type\": \"bloom\",\n";
        ofs << "  \"F\": " << F << ",\n";
        ofs << "  \"model_construct_time_ms\": " << model_construct_time_ms << ",\n";
        ofs << "  \"add_keys_time_ms\": " << add_keys_time_ms << ",\n";
        ofs << "  \"model_size_kb\": " << model_size_kb << "\n";
        ofs << "}\n";
        ofs.close();
    }
}


int main(int argc, char* argv[]) {
    std::string all_pos_key_path;
    std::string X_val_key_path;
    std::string y_val_path;
    std::string bloom_folder_path;
    float F = -1.0;

    int opt;
    while ((opt = getopt(argc, argv, "p:k:y:o:f:")) != -1) {
        switch (opt) {
            case 'p':
                all_pos_key_path = optarg;
                break;
            case 'k':
                X_val_key_path = optarg;
                break;
            case 'y':
                y_val_path = optarg;
                break;
            case 'o':
                bloom_folder_path = optarg;
                break;
            case 'f':
                F = std::atof(optarg);
                break;
            default:
                std::cerr << "Usage: " << argv[0] << " -p <all_pos_key_path> -k <X_val_key_path> -y <y_val_path> -o <bloom_folder_path> -f <F>\n";
                return 1;
        }
    }

    // Check if all required arguments are provided
    if (all_pos_key_path.empty() || X_val_key_path.empty() || y_val_path.empty() || bloom_folder_path.empty() || F < 0) {
        std::cerr << "Usage: " << argv[0] << " -p <all_pos_key_path> -k <X_val_key_path> -y <y_val_path> -o <bloom_folder_path> -f <F>\n";
        return 1;
    }

    // If bloom_model_path already exists, skip constructing the bloom
    std::string bloom_model_path = bloom_folder_path + "/bloom_model.bin";
    if (std::filesystem::exists(bloom_model_path)) {
        std::cout << "Bloom model already exists at " << bloom_model_path << ". Skipping constructing the bloom.\n";
        return 0;
    }

    // Ensure the processing code is executed
    std::vector<std::string> all_pos_key = load_key(all_pos_key_path);
    construct_bloom(all_pos_key, bloom_folder_path, F);

    return 0;
}
