#ifndef BLOOM_H
#define BLOOM_H

#include "../utils/bloom_io.h"
#include <fstream>
#include <iostream>
#include <vector>
#include <string>
#include <map>
#include <stdexcept>
#include <sstream>
#include <cstdio>
#include "bloom.h"

class Bloom {
public:
    Bloom(): bf(16, 16) {
    }
    void configure_bloom(size_t num_keys, float F);
    bloomfilter::Status add(const uint64_t& key);    // Not implemented
    std::vector<bloomfilter::Status> add_all(const std::vector<uint64_t> &keys);
    bloomfilter::Status contains(const uint64_t& key);   // Not implemented
    std::vector<bloomfilter::Status> contains_all(const std::vector<uint64_t> &keys);
    void save_model(const std::string &path);
    void load_model(const std::string &path);

private:
    bloomfilter::BloomFilter<uint64_t, false> bf;
};

void Bloom::configure_bloom(size_t num_keys, float F) {
    int bits_per_item = std::max(1, (int)(-std::log(F) / std::log(2) / std::log(2) + 0.5));
    bloomfilter::BloomFilter<uint64_t, false> bf_(num_keys, bits_per_item);
    this->bf = bf_;
}

bloomfilter::Status Bloom::add(const uint64_t& key) {
    return bf.Add(key);
}

std::vector<bloomfilter::Status> Bloom::add_all(const std::vector<uint64_t> &keys) {
    std::vector<bloomfilter::Status> res;
    res.reserve(keys.size());
    for (size_t i = 0; i < keys.size(); ++i) {
        res.push_back(bf.Add(keys[i]));
    }
    return res;
}

bloomfilter::Status Bloom::contains(const uint64_t& key) {
    return bf.Contain(key);
}

std::vector<bloomfilter::Status> Bloom::contains_all(const std::vector<uint64_t> &keys) {
    std::vector<bloomfilter::Status> predictions;
    predictions.resize(keys.size());
    for (size_t i = 0; i < keys.size(); ++i) {
        predictions[i] = bf.Contain(keys[i]);
    }
    return predictions;
}

void Bloom::save_model(const std::string &path) {
    std::ofstream ofs(path, std::ios::binary);
    if (!ofs) {
        throw std::runtime_error("Failed to open file for saving model: " + path);
    }

    // Save the BloomFilters
    {
        save_bloom_filter(bf, ofs);
    }

    ofs.close();

    std::cout << "[INFO] Model saved to " << path << std::endl;
}

void Bloom::load_model(const std::string &path) {
    if (!std::filesystem::exists(path)) {
        throw std::runtime_error("Model file does not exist: " + path);
    }

    std::ifstream ifs(path, std::ios::binary);
    if (!ifs) {
        throw std::runtime_error("Failed to open file for loading model: " + path);
    }

    // Load the BloomFilters
    {
        bf = load_bloom_filter(ifs);
    }

    ifs.close();

    std::cout << "[INFO] Model loaded from " << path << std::endl;
}

#endif // BLOOM_H
