#ifndef CALIBRATION_MANAGER_H
#define CALIBRATION_MANAGER_H

#include "clbf.h"
#include "burger_config.h"
#include "find_optimal_parameters.h"
#include <xgboost/c_api.h>
#include <vector>
#include <string>
#include <map>
#include <stdexcept>
#include <cmath>
#include <algorithm>
#include <iostream>
#include <fstream>
#include <numeric>


class Calibration_Manager {
public:
    Calibration_Manager();
    Calibration_Manager(const CLBF &clbf);
    void set_clbf(const CLBF &clbf);
    void calibrate(const std::vector<std::vector<float>> &X, const std::vector<std::vector<float>> &y);
    CLBF configure_burger_assembly(long int n, float F, float lambda_, float mu_);
    BurgerConfig get_burger_config() const { return this->burger_config; }

private:
    CLBF clbf;
    std::vector<std::vector<float>> calibration_stats_pos;
    std::vector<std::vector<float>> calibration_stats_neg;
    float calibration_time_pos_us;
    float calibration_time_neg_us;
    BurgerConfig burger_config;
};

Calibration_Manager::Calibration_Manager() {
}

Calibration_Manager::Calibration_Manager(const CLBF &clbf) {
    if (clbf.is_ml_model_trained()) {
        this->clbf = clbf;
    } else {
        throw std::runtime_error("ML model not trained. Please train the model first before setting.");
    }
}

void Calibration_Manager::set_clbf(const CLBF &clbf) {
    if (clbf.is_ml_model_trained()) {
        this->clbf = clbf;
    } else {
        throw std::runtime_error("ML model not trained. Please train the model first before setting.");
    }
}

void Calibration_Manager::calibrate(const std::vector<std::vector<float>> &X, const std::vector<std::vector<float>> &y) {
    if (!this->clbf.is_ml_model_trained()) {
        throw std::runtime_error("ML model not trained. Please train the model first before calibrating.");
    }
    std::vector<std::vector<float>> X_pos;
    std::vector<std::vector<float>> X_neg;
    for (int i = 0; i < (int)X.size(); i++) {
        if (y[i][0] == 1) {
            X_pos.push_back(X[i]);
        } else {
            X_neg.push_back(X[i]);
        }
    }
    auto [csp, ctp] = this->clbf.calibrate(X_pos);
    auto [csn, ctn] = this->clbf.calibrate(X_neg);
    this->calibration_stats_pos = csp;
    this->calibration_stats_neg = csn;
    this->calibration_time_pos_us = ctp;
    this->calibration_time_neg_us = ctn;
}

CLBF Calibration_Manager::configure_burger_assembly(long int n, float F, float lambda_, float mu_) {
    if (!this->clbf.is_ml_model_trained()) {
        throw std::runtime_error("ML model not trained. Please train the model first before configuring.");
    }
    if (this->calibration_stats_pos.size() == 0 || this->calibration_stats_neg.size() == 0) {
        throw std::runtime_error("Calibration data not loaded. Please load the calibration data first before configuring.");
    }
    if (F <= 0 || F >= 1) {
        throw std::invalid_argument("F must be in the range (0, 1).");
    }
    if (lambda_ < 0 || lambda_ > 1) {
        throw std::invalid_argument("lambda_ must be in the range [0, 1].");
    }
    if (mu_ < 0 || mu_ > 1) {
        throw std::invalid_argument("mu_ must be in the range [0, 1].");
    }

    std::cout << "Finding optimal parameters..." << std::endl;
    this->burger_config = find_optimal_parameters(
        n, F, lambda_, mu_, 
        this->calibration_stats_pos, this->calibration_stats_neg, 
        this->calibration_time_pos_us, this->calibration_time_neg_us, 
        this->clbf.xgboost_model_size_kb
    );
    return this->clbf.configure_burger_assembly(n, this->burger_config);
}

#endif // CALIBRATION_MANAGER_H
