#ifndef CALIBRATION_MANAGER_H
#define CALIBRATION_MANAGER_H

#include "disjointadabf.h"
#include "burger_config.h"
#include "find_optimal_parameters.h"
#include <xgboost/c_api.h>
#include <vector>
#include <string>
#include <map>
#include <stdexcept>
#include <cmath>
#include <algorithm>
#include <iostream>
#include <fstream>
#include <numeric>


class Calibration_Manager {
public:
    Calibration_Manager();
    Calibration_Manager(const DisjointAdaBF &disjointadabf);
    void set_disjointadabf(const DisjointAdaBF &disjointadabf);
    void calibrate(const std::vector<std::vector<float>> &X, const std::vector<std::vector<float>> &y);
    DisjointAdaBF configure_burger_assembly(long int n, float F);
    BurgerConfig get_burger_config() const { return this->burger_config; }

private:
    DisjointAdaBF disjointadabf;
    std::vector<std::vector<float>> calibration_stats_pos;
    std::vector<std::vector<float>> calibration_stats_neg;
    float calibration_time_pos_us;
    float calibration_time_neg_us;
    BurgerConfig burger_config;
};

Calibration_Manager::Calibration_Manager() {
}

Calibration_Manager::Calibration_Manager(const DisjointAdaBF &disjointadabf) {
    if (disjointadabf.is_ml_model_trained()) {
        this->disjointadabf = disjointadabf;
    } else {
        throw std::runtime_error("ML model not trained. Please train the model first before setting.");
    }
}

void Calibration_Manager::set_disjointadabf(const DisjointAdaBF &disjointadabf) {
    if (disjointadabf.is_ml_model_trained()) {
        this->disjointadabf = disjointadabf;
    } else {
        throw std::runtime_error("ML model not trained. Please train the model first before setting.");
    }
}

void Calibration_Manager::calibrate(const std::vector<std::vector<float>> &X, const std::vector<std::vector<float>> &y) {
    if (!this->disjointadabf.is_ml_model_trained()) {
        throw std::runtime_error("ML model not trained. Please train the model first before calibrating.");
    }
    std::vector<std::vector<float>> X_pos;
    std::vector<std::vector<float>> X_neg;
    for (int i = 0; i < (int)X.size(); i++) {
        if (y[i][0] == 1) {
            X_pos.push_back(X[i]);
        } else {
            X_neg.push_back(X[i]);
        }
    }
    auto [csp, ctp] = this->disjointadabf.calibrate(X_pos);
    auto [csn, ctn] = this->disjointadabf.calibrate(X_neg);
    this->calibration_stats_pos = csp;
    this->calibration_stats_neg = csn;
    this->calibration_time_pos_us = ctp;
    this->calibration_time_neg_us = ctn;
}

DisjointAdaBF Calibration_Manager::configure_burger_assembly(long int n, float bit_size_of_Ada_BF) {
    if (!this->disjointadabf.is_ml_model_trained()) {
        throw std::runtime_error("ML model not trained. Please train the model first before configuring.");
    }
    if (this->calibration_stats_pos.size() == 0 || this->calibration_stats_neg.size() == 0) {
        throw std::runtime_error("Calibration data not loaded. Please load the calibration data first before configuring.");
    }
    if (bit_size_of_Ada_BF <= this->disjointadabf.xgboost_model_size_kb * 1024 * 8) {
        std::cerr << "Error: Bit size of AdaBF must be greater than the size of the XGBoost model.\n";
        exit(2);
    }

    std::cout << "Finding optimal parameters..." << std::endl;
    this->burger_config = find_optimal_parameters(
        n, bit_size_of_Ada_BF,
        this->calibration_stats_pos, this->calibration_stats_neg, 
        this->calibration_time_pos_us, this->calibration_time_neg_us, 
        this->disjointadabf.xgboost_model_size_kb
    );

    this->burger_config.print();

    return this->disjointadabf.configure_burger_assembly(n, this->burger_config);
}

#endif // CALIBRATION_MANAGER_H
