#ifndef CALIBRATION_MANAGER_H
#define CALIBRATION_MANAGER_H

#include "plbf.h"
#include "burger_config.h"
#include "find_optimal_parameters.h"
#include <xgboost/c_api.h>
#include <vector>
#include <string>
#include <map>
#include <stdexcept>
#include <cmath>
#include <algorithm>
#include <iostream>
#include <fstream>
#include <numeric>


class Calibration_Manager {
public:
    Calibration_Manager();
    Calibration_Manager(const PLBF &plbf);
    void set_plbf(const PLBF &plbf);
    void calibrate(const std::vector<std::vector<float>> &X, const std::vector<std::vector<float>> &y);
    PLBF configure_burger_assembly(long int n, float F);
    BurgerConfig get_burger_config() const { return this->burger_config; }

private:
    PLBF plbf;
    std::vector<std::vector<float>> calibration_stats_pos;
    std::vector<std::vector<float>> calibration_stats_neg;
    float calibration_time_pos_us;
    float calibration_time_neg_us;
    BurgerConfig burger_config;
};

Calibration_Manager::Calibration_Manager() {
}

Calibration_Manager::Calibration_Manager(const PLBF &plbf) {
    if (plbf.is_ml_model_trained()) {
        this->plbf = plbf;
    } else {
        throw std::runtime_error("ML model not trained. Please train the model first before setting.");
    }
}

void Calibration_Manager::set_plbf(const PLBF &plbf) {
    if (plbf.is_ml_model_trained()) {
        this->plbf = plbf;
    } else {
        throw std::runtime_error("ML model not trained. Please train the model first before setting.");
    }
}

void Calibration_Manager::calibrate(const std::vector<std::vector<float>> &X, const std::vector<std::vector<float>> &y) {
    if (!this->plbf.is_ml_model_trained()) {
        throw std::runtime_error("ML model not trained. Please train the model first before calibrating.");
    }
    std::vector<std::vector<float>> X_pos;
    std::vector<std::vector<float>> X_neg;
    for (int i = 0; i < (int)X.size(); i++) {
        if (y[i][0] == 1) {
            X_pos.push_back(X[i]);
        } else {
            X_neg.push_back(X[i]);
        }
    }
    auto [csp, ctp] = this->plbf.calibrate(X_pos);
    auto [csn, ctn] = this->plbf.calibrate(X_neg);
    this->calibration_stats_pos = csp;
    this->calibration_stats_neg = csn;
    this->calibration_time_pos_us = ctp;
    this->calibration_time_neg_us = ctn;
}

PLBF Calibration_Manager::configure_burger_assembly(long int n, float F) {
    if (!this->plbf.is_ml_model_trained()) {
        throw std::runtime_error("ML model not trained. Please train the model first before configuring.");
    }
    if (this->calibration_stats_pos.size() == 0 || this->calibration_stats_neg.size() == 0) {
        throw std::runtime_error("Calibration data not loaded. Please load the calibration data first before configuring.");
    }
    if (F <= 0 || F >= 1) {
        throw std::invalid_argument("F must be in the range (0, 1).");
    }

    std::cout << "Finding optimal parameters..." << std::endl;
    this->burger_config = find_optimal_parameters(
        n, F,
        this->calibration_stats_pos, this->calibration_stats_neg, 
        this->calibration_time_pos_us, this->calibration_time_neg_us, 
        this->plbf.xgboost_model_size_kb
    );
    return this->plbf.configure_burger_assembly(n, this->burger_config);
}

#endif // CALIBRATION_MANAGER_H
