"""
Fused Softmax
=============

In this tutorial, you will write a fused softmax operation that is significantly faster
than PyTorch's native op for a particular class of matrices: those whose rows can fit in
the GPU's SRAM.

In doing so, you will learn about:

* The benefits of kernel fusion for bandwidth-bound operations.

* Reduction operators in Triton.

"""

# %%
# Motivations
# -----------
#
# Custom GPU kernels for elementwise additions are educationally valuable but won't get you very far in practice.
# Let us consider instead the case of a simple (numerically stabilized) softmax operation:

import torch

import triton
import triton.language as tl
from triton.runtime import driver

# %%
# When implemented naively in PyTorch, computing :code:`y = naive_softmax(x)` for :math:`x \in R^{M \times N}`
# requires reading :math:`5MN + 2M` elements from DRAM and writing back :math:`3MN + 2M` elements.
# This is obviously wasteful; we'd prefer to have a custom "fused" kernel that only reads
# X once and does all the necessary computations on-chip.
# Doing so would require reading and writing back only :math:`MN` bytes, so we could
# expect a theoretical speed-up of ~4x (i.e., :math:`(8MN + 4M) / 2MN`).
# The `torch.jit.script` flags aims to perform this kind of "kernel fusion" automatically
# but, as we will see later, it is still far from ideal.

# %%
# Compute Kernel
# --------------
#
# Our softmax kernel works as follows: each program loads a set of rows of the input matrix X strided by number of programs,
# normalizes it and writes back the result to the output Y.
#
# Note that one important limitation of Triton is that each block must have a
# power-of-two number of elements, so we need to internally "pad" each row and guard the
# memory operations properly if we want to handle any possible input shapes:


@triton.jit
def tril_softmax_kernel(y_ptr,
                        x_ptr,
                        x_row_stride,
                        y_row_stride,
                        B, C, T,
                        scale: float,
                        BLOCK_SIZE: tl.constexpr,
                        num_stages: tl.constexpr):
    # starting row of the program
    row_start = tl.program_id(0)
    row_step = tl.num_programs(0)
    for row_idx in tl.range(row_start, B * C, row_step, num_stages=num_stages):
        # The stride represents how much we need to increase the pointer to advance 1 row
        row_start_ptr = x_ptr + row_idx * x_row_stride
        # The block size is the next power of two greater than n_cols, so we can fit each
        # row in a single block
        col_offsets = tl.arange(0, BLOCK_SIZE)
        x_ptrs = row_start_ptr + col_offsets
        # Load the row into SRAM, using a mask since BLOCK_SIZE may be > than n_cols
        mask = col_offsets < T - C + (row_idx % C)
        row = tl.load(x_ptrs, mask=mask, other=-float('inf'))
        # Subtract maximum for numerical stability
        row_minus_max = row - tl.max(row, axis=0)
        # Note that exponentiation in Triton is fast but approximate (i.e., think __expf in CUDA)
        numerator = tl.exp(row_minus_max * scale)
        denominator = tl.sum(numerator, axis=0)
        softmax_output = numerator / denominator
        # Write back output to DRAM
        y_row_start_ptr = y_ptr + row_idx * y_row_stride
        y_ptrs = y_row_start_ptr + col_offsets
        tl.store(y_ptrs, softmax_output, mask=mask)


# %%
# We can create a helper function that enqueues the kernel and its (meta-)arguments for any given input tensor.

properties = driver.active.utils.get_device_properties(0)
NUM_SM = properties["multiprocessor_count"]
NUM_REGS = properties["max_num_regs"]
SIZE_SMEM = properties["max_shared_mem"]
WARP_SIZE = properties["warpSize"]
target = triton.runtime.driver.active.get_current_target()
kernels = {}


def tril_softmax(x, scale=1.):
    xshp = x.shape
    x = x.flatten(0, -3)
    B, C, T = x.shape

    # The block size of each loop iteration is the smallest power of two greater than the number of columns in `x`
    BLOCK_SIZE = triton.next_power_of_2(T)

    # Another trick we can use is to ask the compiler to use more threads per row by
    # increasing the number of warps (`num_warps`) over which each row is distributed.
    # You will see in the next tutorial how to auto-tune this value in a more natural
    # way so you don't have to come up with manual heuristics yourself.
    num_warps = 8

    # Number of software pipelining stages.
    num_stages = 4 if SIZE_SMEM > 200000 else 2

    # Allocate output
    y = torch.empty_like(x)

    kernel = tril_softmax_kernel.warmup(y, x,
                                        x.stride(1), y.stride(1), B, C, T, scale,
                                        BLOCK_SIZE=BLOCK_SIZE,
                                        num_stages=num_stages,
                                        num_warps=num_warps,
                                        grid=(1, ))
    kernel._init_handles()
    if False:
        # pre-compile kernel to get register usage and compute thread occupancy.
        n_regs = kernel.n_regs
        size_smem = kernel.metadata.shared
        occupancy = NUM_REGS // (n_regs * WARP_SIZE * num_warps)
        occupancy = min(occupancy, SIZE_SMEM // size_smem)
        num_programs = NUM_SM * occupancy
        num_programs = min(num_programs * 4, B * C)
    else:
        num_programs = B * C

    # Create a number of persistent programs.
    kernel[(num_programs, 1, 1)](y, x, x.stride(1), y.stride(1), B, C, T, scale)
    return y.view(xshp)


def naive_tril_softmax(x: torch.Tensor):
    diag = x.size(-1) - x.size(-2) - 1
    xmax, _ = x.tril().max(dim=-1, keepdim=True)
    exp = (x - xmax).exp().tril(diag)
    expsum = exp.sum(dim=-1, keepdim=True)
    expsum[expsum == 0] = 1e-5
    return exp / expsum


def naive_tril_softmax_2(x: torch.Tensor):
    diag = x.size(-1) - x.size(-2) - 1
    xmax, _ = x.tril().max(dim=-1, keepdim=True)
    exp = (x - xmax).exp2().tril(diag)
    expsum = exp.sum(dim=-1, keepdim=True)
    expsum[expsum == 0] = 1e-5
    return exp / expsum



if __name__ == '__main__':
    # B, C, T = 32 * 2, 128, 4096
    B, C, T = 4, 8, 16
    x = torch.randn(B, C, T, dtype=torch.bfloat16, device='cuda')
    y = tril_softmax(x)
    print(y)
