function [w, error_history] = batchedOja(X, w0, eta, v1, B)
% batchedOja  Batched implementation of Oja's algorithm without forming
%             the full covariance matrix.
%
%   For batch b (size = m):
%       z   = X_b * w              % m × 1 products
%       g   = (X_b' * z) / m       % d × 1 gradient (C_b * w)
%       w   = w + η g;  w = w/‖w‖  % update & renormalise
%
%   INPUTS / OUTPUTS are identical to the earlier version.
% -------------------------------------------------------------------------

[T, ~] = size(X);
if ~(isscalar(B) && B >= 1 && B <= T && B == floor(B))
    error('B must be an integer between 1 and T.');
end

w              = w0(:);            % ensure column vector
autoNorm       = @(v) v / norm(v); % inline for speed
error_history  = zeros(B,1);

batchSize      = floor(T / B);
remainder      = mod(T, B);
startIdx       = 1;

for b = 1:B
    % Determine batch bounds
    m        = batchSize + (b <= remainder);
    endIdx   = startIdx + m - 1;

    Xi       = X(startIdx:endIdx, :);   % m × d

    % Efficient gradient: g = (1/m) * X_b' * (X_b * w)
    z        = Xi * w;                  % m × 1
    g        = (Xi' * z) / m;           % d × 1  (empirical C_b * w)

    % Update and normalise
    w        = autoNorm(w + eta * g);

    % Record sin² error
    error_history(b) = 1 - (abs(w' * v1))^2;

    startIdx = endIdx + 1;              % advance pointer
end
end
