function sr = mantissaRound(x, delta, alpha)
% stochasticRoundExp  Stochastically round each entry of x to the nearest
%                     points on an exponential-spacing grid.
%
%   Grid definition (for k = 0,1,2,… and mirrored for negatives):
%       q_k + delta/alpha = (1+alpha)^k * (delta/alpha)
%       ⇒ q_k = (delta/alpha) * ( (1+alpha)^k − 1 )
%
%   PARAMETERS
%       x      – scalar, vector, or matrix of real numbers
%       delta  – base resolution (> 0)
%       alpha  – growth factor (> 0)  (common choice ~ 2^-m for m-bit mantissa)
%
%   RETURNS
%       sr     – stochastically-rounded values, same size as x
%
%   METHOD
%       • For each |x_i|, find the two adjacent grid points q_k ≤ |x_i| ≤ q_{k+1}.
%       • Round to the lower point with probability (q_{k+1}−|x_i|)/(q_{k+1}−q_k)
%         and to the upper point with the complementary probability.
%       • Preserve the sign of x_i.

% -------------------------------------------------------------------------
% Pre-compute constants
% -------------------------------------------------------------------------
base   = delta / alpha;      % = q_0 + delta/alpha
ratio  = 1 + alpha;          % geometric ratio between successive buckets
absx   = abs(x);
sgn    = sign(x);

% Handle x == 0 quickly
sr = zeros(size(x));         % initialise output
nonZeroMask = absx > 0;      % logical mask of non-zero entries
if ~any(nonZeroMask), return; end

absx_nz = absx(nonZeroMask);

% -------------------------------------------------------------------------
% Locate bucket indices (vectorised)
%   Solve for k:  q_k ≤ |x| < q_{k+1}
%   q_k = base * (ratio^k − 1)
%   ⇒ |x| + base = base * ratio^k
%   ⇒ k = floor( log_ratio( |x|/base + 1 ) )
% -------------------------------------------------------------------------
k = floor( log( absx_nz / base + 1 ) / log(ratio) );

% Lower/upper magnitudes
lowerMag = base .* (ratio.^k       - 1);   % q_k
upperMag = base .* (ratio.^(k + 1) - 1);   % q_{k+1}

% -------------------------------------------------------------------------
% Stochastic rounding
% -------------------------------------------------------------------------
% Probability of rounding *up* to q_{k+1}
probUp = (absx_nz - lowerMag) ./ (upperMag - lowerMag);

% Draw uniform random numbers
r = rand(size(probUp));

% Allocate result for non-zero entries
sr_nz = lowerMag;                 % default: round down
sr_nz(r < probUp) = upperMag(r < probUp);   % round up where needed

% Restore signs and write back
sr(nonZeroMask) = sgn(nonZeroMask) .* sr_nz;
end