function [w, error_history] = batchedOja_lowPrecision( ...
    X, w0, eta, v1, quantFn, delta, alpha, numBatches)
% batchedOja_lowPrecision  B-batch Oja’s rule with per-batch quantization.
%
% USAGE:
%   [w, errs] = batchedOja_lowPrecision( ...
%      X, w0, eta, v1, quantFn, delta, alpha, B );
%
% INPUTS:
%   X          – (n×d) data matrix
%   w0         – (d×1) initial unit vector
%   eta        – learning rate (already scaled if desired)
%   v1         – true top eigenvector (for error)
%   quantFn    – handle @(z,δ) or @(z,δ,α)
%   delta      – quant step
%   alpha      – mantissa growth (empty for uniform)
%   numBatches – # of consecutive minibatches
%
% OUTPUTS:
%   w             – final unit‐norm estimate
%   error_history – (B×1) sin²‐errors per batch

[n, ~] = size(X);
if numBatches < 1 || numBatches~=floor(numBatches)
    error('numBatches must be a positive integer.');
end

% partition sizes
base   = floor(n/numBatches);
extra  = mod(n, numBatches);
w      = w0(:) / norm(w0);
error_history = zeros(numBatches,1);
startIdx = 1;

for b = 1:numBatches
    % batch bounds
    m      = base + (b <= extra);
    endIdx = startIdx + m - 1;
    Xi     = X(startIdx:endIdx, :);   % (m×d)

    % quantize previous iterate
    wq = quantHelper(quantFn, w, delta, alpha);

    % per‐sample rank-1 updates with quant
    %   accumulate Σ_j Quant( xi (xiᵀ wq), δ )
    updates = zeros(size(w));
    for j = 1:m
        xj = Xi(j,:).';
        pj = xj.' * wq;
        pj = quantHelper(quantFn, pj, delta, alpha);
        uj = pj * xj;
        uj = quantHelper(quantFn, uj, delta, alpha);
        updates = updates + uj;
    end

    % scale, quant, and apply
    g = updates / m;
    y = eta * g;
    y = quantHelper(quantFn, y, delta, alpha);

    u = wq + y;
    u = u / norm(u);
    w = quantHelper(quantFn, u, delta, alpha);

    w_prime = w/norm(w);

    % record error
    error_history(b) = 1 - ( (abs(w_prime' * v1).^2) );
    startIdx = endIdx + 1;
end
end