function [X, Sigma, v1, lambda1, lambda2] = generateData(n, d, beta)
% generateData generates n data points in R^d following the procedure in Section 4 
% ("Experimental validation of the online multiplier bootstrap") of :contentReference[oaicite:0]{index=0}.
%
% The process is as follows:
%   1. Generate Z, an n×d matrix with IID entries from Uniform(-sqrt(3), sqrt(3))
%      (ensuring each coordinate has unit variance).
%   2. Construct a d×d kernel matrix K with entries K(i,j) = exp(-abs(i-j)^0.01).
%   3. Define a variance vector: sigma(i) = 5 * i^(-beta).
%   4. Build the covariance matrix Σ = (sigma * sigma') .* K.
%   5. Compute Σ^(1/2) using eigendecomposition and set X = (Σ^(1/2))*Z'.
%   6. Compute the top eigenvector and eigenvalues of Σ.
%
% INPUTS:
%   n    - Number of data points.
%   d    - Data dimension.
%   beta - Variance decay parameter (e.g., 0.2 or 1).
%
% OUTPUTS:
%   X       - An n×d matrix where each row is a sample with covariance Σ.
%   Sigma   - The d×d covariance matrix used.
%   v1      - The top eigenvector of Σ.
%   lambda1 - The largest eigenvalue of Σ.
%   lambda2 - The second largest eigenvalue of Σ.

% Set constant for the kernel matrix
c = 0.01;

% Step 1: Generate Z with entries from Uniform(-sqrt(3), sqrt(3))
a = sqrt(3);
Z = -a + 2 * a * rand(n, d);

% Step 2: Construct the kernel matrix K (d×d) with K(i,j) = exp(-abs(i-j)^c)
K = zeros(d, d);
for i = 1:d
    for j = 1:d
        K(i, j) = exp(-abs(i - j)^c);
    end
end

% Step 3: Define the variance vector: sigma(i) = 5 * i^(-beta)
sigma_vec = 5 * ((1:d) .^ (-beta))';

% Step 4: Build the covariance matrix Σ with Σ(i,j) = K(i,j) * sigma(i) * sigma(j)
Sigma = (sigma_vec * sigma_vec') .* K;

% Step 5: Compute the square root of Σ using eigendecomposition
[V, D] = eig(Sigma);
% Ensure nonnegative eigenvalues (Σ is PSD by construction)
D = max(D, 0);
Sigma_sqrt = V * sqrt(D) * V';

% Generate data: X = (Σ^(1/2) * Z')', so that each row is a sample.
X = (Sigma_sqrt * Z')';

% Step 6: Compute eigen-decomposition of Σ for the ground truth
[VSigma, DSigma] = eig(Sigma);
eigvals = diag(DSigma);
[sortedEigvals, idx] = sort(eigvals, 'descend');
v1 = VSigma(:, idx(1));
lambda1 = sortedEigvals(1);
if d >= 2
    lambda2 = sortedEigvals(2);
else
    lambda2 = sortedEigvals(1);
end
end