function [w, error_history] = regularOja(X, w0, eta, v1)
% regularOja implements the standard Oja's algorithm with normalization.
%
% INPUTS:
%   X             - T x d data matrix (each row is a data point).
%   w0            - Initial weight vector (d x 1, unit norm).
%   eta           - Learning rate.
%   v1            - True top eigenvector (for error computation).
%
% OUTPUTS:
%   w             - Final estimated eigenvector.
%   error_history - Vector of sin^2 error at each iteration.
%
% Update rule:
%   w = w + eta*(w' * x_t)* x_t, then normalize w.

[T, ~] = size(X);
w = w0;
error_history = zeros(T,1);

for t = 1:T
    x_t = X(t,:)';
    % Compute inner product
    y = w' * x_t;
    % Update (unnormalized)
    w = w + eta * y * x_t;
    % Normalize to unit norm
    w = w / norm(w);
    % Compute sin^2 error = 1 - (|<w, v1>|^2)
    error_history(t) = 1 - (abs(w' * v1))^2;
end

end
