function [w, error_history] = regularOja_lowPrecision( ...
    X, w0, eta, v1, quantFn, delta, alpha)
% regularOja_lowPrecision  Oja’s algorithm with per-sample quantization.
%
% USAGE:
%   [w, errs] = regularOja_lowPrecision( ...
%        X, w0, eta, v1, quantFn, delta, alpha );
%
% INPUTS:
%   X       – (n×d) data matrix, rows are samples
%   w0      – (d×1) initial unit‐norm vector
%   eta     – learning rate
%   v1      – (d×1) true top eigenvector (only for error logging)
%   quantFn – function handle: @(z,delta) or @(z,delta,alpha)
%   delta   – quantization step size
%   alpha   – growth factor (empty for uniform quant)
%
% OUTPUTS:
%   w             – final unit‐norm estimate of eigenvector
%   error_history – (n×1) sin²‐error sequence

[n, d] = size(X);
w = w0(:) / norm(w0);
error_history = zeros(n,1);

for t = 1:n
    x = X(t,:).';
    % quantize current iterate
    w = quantHelper(quantFn, w, delta, alpha);

    % inner product (quantized)
    p = w.' * x;
    p = quantHelper(quantFn, p, delta, alpha);

    % rank-1 update (quantized)
    u = eta * p * x;
    u = quantHelper(quantFn, u, delta, alpha);

    % accumulate & quantize
    w = w + u;
    w = quantHelper(quantFn, w, delta, alpha);

    % normalize & quantize
    w = w / norm(w);
    w = quantHelper(quantFn, w, delta, alpha);

    w_prime = w/norm(w);

    % record sin^2‐error
    error_history(t) = 1 - ( (abs(w_prime' * v1).^2) );
end
end