from gym.spaces import Space
import numpy as np
from typing import Union, Optional

from models.action_dist import ActionDistribution
from models.modelv2 import ModelV2
from utils.annotations import override
from utils.exploration.exploration import Exploration
from utils.exploration.random import Random
from utils.framework import (
    try_import_torch,
    get_variable,
    TensorType,
)
from utils.numpy import convert_to_numpy
from utils.schedules import Schedule
from utils.schedules.piecewise_schedule import PiecewiseSchedule

torch, _ = try_import_torch()


class GaussianNoise(Exploration):
    """An exploration that adds white noise to continuous actions.

    If explore=True, returns actions plus scale (annealed over time) x
    Gaussian noise. Also, some completely random period is possible at the
    beginning.

    If explore=False, returns the deterministic action.
    """

    def __init__(
        self,
        action_space: Space,
        *,
        framework: str,
        model: ModelV2,
        random_timesteps: int = 1000,
        stddev: float = 0.1,
        initial_scale: float = 1.0,
        final_scale: float = 0.02,
        scale_timesteps: int = 10000,
        scale_schedule: Optional[Schedule] = None,
        **kwargs
    ):
        """Initializes a GaussianNoise instance.

        Args:
            random_timesteps: The number of timesteps for which to act
                completely randomly. Only after this number of timesteps, the
                `self.scale` annealing process will start (see below).
            stddev: The stddev (sigma) to use for the
                Gaussian noise to be added to the actions.
            initial_scale: The initial scaling weight to multiply
                the noise with.
            final_scale: The final scaling weight to multiply
                the noise with.
            scale_timesteps: The timesteps over which to linearly anneal
                the scaling factor (after(!) having used random actions for
                `random_timesteps` steps).
            scale_schedule: An optional Schedule object
                to use (instead of constructing one from the given parameters).
        """
        assert framework is not None
        super().__init__(action_space, model=model, framework=framework, **kwargs)

        # Create the Random exploration module (used for the first n
        # timesteps).
        self.random_timesteps = random_timesteps
        self.random_exploration = Random(
            action_space, model=self.model, framework=self.framework, **kwargs
        )

        self.stddev = stddev
        # The `scale` annealing schedule.
        self.scale_schedule = scale_schedule or PiecewiseSchedule(
            endpoints=[
                (random_timesteps, initial_scale),
                (random_timesteps + scale_timesteps, final_scale),
            ],
            outside_value=final_scale,
            framework=self.framework,
        )

        # The current timestep value.
        self.last_timestep = get_variable(
            np.array(0, np.int64),
            framework=self.framework,
            dtype=np.int64,
        )

    @override(Exploration)
    def get_exploration_action(
        self,
        *,
        action_distribution: ActionDistribution,
        timestep: Union[int, TensorType],
        explore: bool = True
    ):
        # Adds IID Gaussian noise for exploration, TD3-style.
        return self._get_torch_exploration_action(
            action_distribution, explore, timestep
        )

    def _get_torch_exploration_action(
        self,
        action_dist: ActionDistribution,
        explore: bool,
        timestep: Union[int, TensorType],
    ):
        # Set last timestep or (if not given) increase by one.
        self.last_timestep = (
            timestep if timestep is not None else self.last_timestep + 1
        )

        # Apply exploration.
        if explore:
            # Random exploration phase.
            if self.last_timestep < self.random_timesteps:
                action, _ = self.random_exploration.get_torch_exploration_action(
                    action_dist, explore=True
                )
            # Take a Gaussian sample with our stddev (mean=0.0) and scale it.
            else:
                det_actions = action_dist.deterministic_sample()
                scale = self.scale_schedule(self.last_timestep)
                gaussian_sample = scale * torch.normal(
                    mean=torch.zeros(det_actions.size()), std=self.stddev
                ).to(self.device)
                action = torch.min(
                    torch.max(
                        det_actions + gaussian_sample,
                        torch.tensor(
                            self.action_space.low,
                            dtype=torch.float32,
                            device=self.device,
                        ),
                    ),
                    torch.tensor(
                        self.action_space.high, dtype=torch.float32, device=self.device
                    ),
                )
        # No exploration -> Return deterministic actions.
        else:
            action = action_dist.deterministic_sample()

        # Logp=always zero.
        logp = torch.zeros((action.size()[0],), dtype=torch.float32, device=self.device)

        return action, logp

    @override(Exploration)
    def get_state(self):
        """Returns the current scale value.

        Returns:
            float: The current scale value.
        """
        scale = self.scale_schedule(self.last_timestep)
        return {
            "cur_scale": convert_to_numpy(scale),
            "last_timestep": convert_to_numpy(self.last_timestep),
        }

    @override(Exploration)
    def set_state(self, state: dict) -> None:
        if isinstance(self.last_timestep, int):
            self.last_timestep = state["last_timestep"]
        else:
            self.last_timestep.assign(state["last_timestep"])
