import torch
import torch.optim as optim
import torchvision
import random
import math
import torch.nn as nn
import torch.nn.functional as F
from torchvision import datasets, transforms
from torch.autograd import Variable
from enum import IntEnum
from collections import OrderedDict

# Experiment 1: FFNN for MNIST, FMNIST

class FFNN(nn.Module):
    
    def __init__(self, args):
        
        print("FFNN!")
        print("layers:", args["layer_size"])
        self.layers = args["layer_size"]
        self.acts = args["acts"]
        self.nlayer = len(self.layers)-1
        super(FFNN, self).__init__()
        
        linear_layers = []
        bn_layers = []
        
        for i in range(self.nlayer):
            linear_layers.append(nn.Linear(self.layers[i], self.layers[i+1]))
            bn_layers.append(nn.BatchNorm1d(num_features = self.layers[i+1]))
            
        self.linear_layer = nn.ModuleList(linear_layers)
        self.bn_layers = nn.ModuleList(bn_layers)
    
    def activate(self, activation):
        
        if activation == "relu":
            act_f = nn.ReLU()
        if activation == "softmax":
            act_f = nn.Softmax()
        if activation == "sigmoid":
            act_f = nn.Sigmoid()
        if activation == "elu":    
            act_f = nn.ELU()
            
        return act_f
        
    def forward(self, x):
        
        # print("x.size:", x.size())
        x = x.view(-1, self.layers[0])
        out = x
        
        for i in range(self.nlayer):
            next_layer = self.linear_layer[i]
            # next_bn_layer = self.bn_layers[i]
            out = next_layer(out)
            # out = next_bn_layer(out)
            if self.acts[i] != "linear":
                # print("self.acts[i]", self.acts[i])
                act_func = self.activate(self.acts[i])
                out = act_func(out)
            # print("out.size", x.size())
                
        return out
    
# Experiment 2: Conv AE: MNIST, FMNIST 

class conv_autoencoder(nn.Module):
    def __init__(self, args):
        super(conv_autoencoder, self).__init__()
        print("conv_autoencoder")
        self.noise = args["noise_level"]
        self.flexible_act = args["flexible_act"]
        
        if self.flexible_act == True or self.flexible_act == "FReLu_reg":
            print("FReLu")
            self.p_act_layer_1 = p_act_layer(input_features = 3, args = args)
            self.p_act_layer_2 = p_act_layer(input_features = 5, args = args)
        elif self.flexible_act == "PReLu":
            print("PReLu")
            self.p_act_layer_1 = nn.PReLU()
            self.p_act_layer_2 = nn.PReLU()
            self.p_act_layer_3 = nn.PReLU()
            self.p_act_layer_4 = nn.PReLU()
        elif self.flexible_act == "PReLu_reg":
            self.p_act_layer_1 = p_act_layer_prelu(12)
            self.p_act_layer_2 = p_act_layer_prelu(12)
            self.p_act_layer_3 = p_act_layer_prelu(12)
            self.p_act_layer_4 = p_act_layer_prelu(12)
        elif self.flexible_act == "Relu_Elu" or self.flexible_act == "Relu_Elu_reg" or self.flexible_act == "Relu_Elu_reg_1":
            self.p_act_layer_1 = p_act_layer_relu_elu(input_features = 3)
            self.p_act_layer_2 = p_act_layer_relu_elu(input_features = 5)
            self.p_act_layer_3 = p_act_layer_relu_elu(input_features = 10)
            self.p_act_layer_4 = p_act_layer_relu_elu(input_features = 15)
        elif self.flexible_act == "Gelu":
            print("Gelu")
            self.p_act_layer_1 = GELU()
            self.p_act_layer_2 = GELU()
            self.p_act_layer_3 = GELU()
            self.p_act_layer_4 = GELU()
        elif self.flexible_act == "Elu":
            print("Elu")
            self.p_act_layer_1 = nn.ELU()
            self.p_act_layer_2 = nn.ELU()
            self.p_act_layer_3 = nn.ELU()
            self.p_act_layer_4 = nn.ELU()
        else:
            print("False")
            self.p_act_layer_1 = nn.ReLU(True)
            self.p_act_layer_2 = nn.ReLU(True)
            self.p_act_layer_3 = nn.ReLU(True)
            self.p_act_layer_4 = nn.ReLU(True)
            
        if args["dataset"]=="CIFAR10":
            dim = 3
        if args["dataset"]=="MNIST" or args["dataset"]=="FMNIST":
            dim = 1
        
        self.encoder = nn.Sequential(
            nn.Conv2d(dim, 16, 3, stride=3, padding=1),  # b, 16, 10, 10
            # nn.ReLU(True),
            self.p_act_layer_3,
            nn.MaxPool2d(2, stride=2),  # b, 16, 5, 5
            nn.Conv2d(16, 8, 3, stride=2, padding=1),  # b, 8, 3, 3
            # nn.ReLU(True),
            self.p_act_layer_1,
            nn.MaxPool2d(2, stride=1)  # b, 8, 2, 2
        )
        self.decoder = nn.Sequential(
            nn.ConvTranspose2d(8, 16, 3, stride=2),  # b, 16, 5, 5
            # nn.ReLU(True),
            self.p_act_layer_2,
            nn.ConvTranspose2d(16, 8, 5, stride=3, padding=1),  # b, 8, 15, 15
            # nn.ReLU(True),
            self.p_act_layer_4,
            nn.ConvTranspose2d(8, dim, 2, stride=2, padding=1),  # b, 1, 28, 28
            nn.Tanh()
        )

    def forward(self, x, x_n = None):
        
        if self.noise == None:
            x = self.encoder(x)
        else:
            x = self.encoder(x_n)
        x = self.decoder(x)
        
        return x

class LeNet_5(nn.Module):
    def __init__(self, args):
        super(LeNet_5, self).__init__()
        print("lenet-5!!!")
        self.new_acts = args["flexible_act"]
        if args["dataset"] == "MNIST" or args["dataset"] == "FMNIST":
            # print("MNIST")
            self.conv1 = nn.Conv2d(1, 6, 5)
        if args["dataset"] == "CIFAR10" or args["dataset"] == "SVHN":
            # print("CIFAR10")
            self.conv1 = nn.Conv2d(3, 6, 5)
        self.conv2 = nn.Conv2d(6, 16, 5)
        self.fc1   = nn.Linear(16*5*5, 120)
        self.fc2   = nn.Linear(120, 84)
        self.fc3   = nn.Linear(84, 10)
        self.bn_layers_1 = nn.BatchNorm1d(120)
        self.bn_layers_2 = nn.BatchNorm1d(84)
        
        if self.new_acts == False:
            self.p_act_layer_1 = nn.ReLU()
            self.p_act_layer_2 = nn.ReLU()
        else:
            print("new_act!")
            self.p_act_layer_1 = nn.ReLU()
            # self.p_act_layer_1 = p_act_layer(120, args)
            self.p_act_layer_2 = p_act_layer(84, args)

    def forward(self, x):
        out = F.relu(self.conv1(x))
        out = F.max_pool2d(out, 2)
        out = F.relu(self.conv2(out))
        out = F.max_pool2d(out, 2)
        out = out.view(out.size(0), -1)
        # out = F.relu(self.fc1(out))
        # out = F.relu(self.fc2(out))
        out = self.p_act_layer_1(self.bn_layers_1(self.fc1(out)))
        out = self.p_act_layer_2(self.bn_layers_2(self.fc2(out)))
    
        out = self.fc3(out)
        return out
    
class L4_Net(nn.Module):
    def __init__(self):
        super(L4_Net, self).__init__()
        self.conv1 = nn.Conv2d(1, 10, kernel_size=5)
        self.conv2 = nn.Conv2d(10, 20, kernel_size=5)
        self.conv2_drop = nn.Dropout2d()
        self.fc1 = nn.Linear(320, 50)
        self.fc2 = nn.Linear(50, 10)

    def forward(self, x):
        x = F.relu(F.max_pool2d(self.conv1(x), 2))
        x = F.relu(F.max_pool2d(self.conv2_drop(self.conv2(x)), 2))
        x = x.view(-1, 320)
        x = F.relu(self.fc1(x))
        x = F.dropout(x, training=self.training)
        x = self.fc2(x)
        return F.log_softmax(x)
    
# Experiment 3: ResNet: CIFAR10, SVHN


class BasicBlock(nn.Module):

    expansion = 1
    def __init__(self, in_planes, planes, stride=1):

        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.shortcut = nn.Sequential()

        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)

        return out


class Bottleneck(nn.Module):

    expansion = 4
    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion *
                               planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)
        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)

        return out

class ResNet(nn.Module):

    def __init__(self, block, num_blocks, num_classes=10):
        super(ResNet, self).__init__()
        self.in_planes = 64
        self.conv1 = nn.Conv2d(3, 64, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.linear = nn.Linear(512*block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []

        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)

        return out

# Resnet with definations of image preprocessing modules, convolution filter and Residual Block

# Define 3x3 convolution as a new function since we will use it for multiple times in building Residual Block

def conv3x3(in_channels, out_channels, stride=1):
    return nn.Conv2d(in_channels, out_channels, kernel_size=3, 
                     stride=stride, padding=1, bias=False)

# Residual block
class ResidualBlock(nn.Module):
    def __init__(self, in_channels, out_channels, stride=1, downsample=None):
        super(ResidualBlock, self).__init__()
        self.conv1 = conv3x3(in_channels, out_channels, stride)
        self.bn1 = nn.BatchNorm2d(out_channels)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(out_channels, out_channels)
        self.bn2 = nn.BatchNorm2d(out_channels)
        self.downsample = downsample
        
    def forward(self, x):
        residual = x
        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)
        out = self.conv2(out)
        out = self.bn2(out)
        if self.downsample:
            residual = self.downsample(x)
            
        out += residual # With identity mapping by shortcuts
        out = self.relu(out)
        return out

# ResNet
class ResNet_0(nn.Module):
    def __init__(self, block, layers, num_classes=10):
        super(ResNet, self).__init__()
        print("Resnet!")
        self.in_channels = 16
        self.conv = conv3x3(3, 16)
        self.bn = nn.BatchNorm2d(16)
        self.relu = nn.ReLU(inplace=True)
        self.layer1 = self.make_layer(block, 16, layers[0])
        self.layer2 = self.make_layer(block, 32, layers[1], 2)
        self.layer3 = self.make_layer(block, 64, layers[2], 2)
        self.avg_pool = nn.AvgPool2d(8)
        self.fc = nn.Linear(64, num_classes)
        
    # Return a sequence of layers built by ResidualBlocks
    def make_layer(self, block, out_channels, blocks, stride=1):
        downsample = None
        if (stride != 1) or (self.in_channels != out_channels):
            downsample = nn.Sequential(
                conv3x3(self.in_channels, out_channels, stride=stride),
                nn.BatchNorm2d(out_channels))
        layers = []
        layers.append(block(self.in_channels, out_channels, stride, downsample))
        self.in_channels = out_channels
        for i in range(1, blocks):
            layers.append(block(out_channels, out_channels))
        return nn.Sequential(*layers)
    
    # The forward flow for ResNet
    def forward(self, x):
        # print("resnet forward")
        out = self.conv(x)
        out = self.bn(out)
        out = self.relu(out)
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.avg_pool(out)
        out = out.view(out.size(0), -1)
        out = self.fc(out)
        return out













