import numpy as np
from sklearn.neighbors import NearestNeighbors


def authenticity_score(real_data: np.ndarray, syn_data: np.ndarray) -> float:
    """ compute the authenticity as for https://arxiv.org/pdf/2102.08921 :
      Authenticity measures the fraction of synthetic samples that are invented by the model and not copied from the training data.
    """

    nbrs_real = NearestNeighbors(n_neighbors=2, n_jobs=-1, p=2).fit(real_data)
    real_to_real, _ = nbrs_real.kneighbors(real_data)

    nbrs_synth = NearestNeighbors(n_neighbors=1, n_jobs=-1, p=2).fit(syn_data)
    real_to_synth, real_to_synth_args = nbrs_synth.kneighbors(real_data)

    # Let us find the closest real point to any real point, excluding itself (therefore 1 instead of 0)
    real_to_real = real_to_real[:, 1].squeeze()
    real_to_synth = real_to_synth.squeeze()
    real_to_synth_args = real_to_synth_args.squeeze()

    # compute the actual authenticity for each point
    authen = real_to_real[real_to_synth_args] < real_to_synth
    authenticity = np.mean(authen)

    # the higher, the better
    return authenticity
