#! python

import csv
import math
import os
import matplotlib.pyplot as plt
from pathlib import Path

evs = 4000

def script_dir():
    spath =  Path(os.path.abspath(__file__))
    sdir = spath.parent
    return str(sdir)

def root_dir():
    spath =  Path(os.path.abspath(__file__))
    sdir = spath.parent
    rdir = sdir.parent
    return str(rdir)


def get_progress_csv_files(directory, funcname, algname):
    rs = []
    token = '_'+algname+'_'
    ftoken = '_'+funcname
    for fn in os.listdir(directory):
        if token in fn and ftoken in fn and fn.endswith('.csv'):
            fpath = os.path.join(directory, fn)
            rs.append(fpath)
            #print('\t\t->%s '%fn)
    print('\t..%d csv files found for [%s] in directory: %s '%(len(rs), algname, directory))
    return rs



class AlgoData:
    def __init__(self, algname, funcname, csvlist, perc = '0.95', reverse_sign=False):
        self.alg = algname
        self.func = funcname
        self.prog_data = []
        
        self.avg=None
        self.std=None
        self.conf_itv = 0.0
        print (':%d files for %s with %s...'%(len(csvlist),funcname, algname))
        for csvfn in csvlist:
            apath = []
            print('opening: %s ...'%csvfn)
            with open(csvfn) as csvdata:
                csvrdr = csv.reader(csvdata, delimiter=',')
                for row in csvrdr:
                    v = float(row[0])
                    if reverse_sign:
                        v = -v
                    apath.append(v)
                    if len(apath) >= evs:
                        break

            print('\t csv file [%s] : %d records '%(csvfn, len(apath)))
            if len(apath)>0 and len(apath)<evs:
                lv = apath[-1]
                for i in range(evs-len(apath)):
                    apath.append(lv)
                print('\t\tappending to %d records'%evs)
            self.prog_data.append(apath)

        self._calc(perc = perc)

    def _calc_ci(self, perc, mean, stdev, n):
    	#ref https://www.mathsisfun.com/data/confidence-interval.html
        Z = 0.0
        if perc == '0.95':
            Z = 1.960
        elif perc == '0.99':
            Z = 2.576
        elif perc == '0.995':
            Z = 2.807
        elif perc == '0.999':
            Z = 3.291
        else:
            raise ValueError('Undefined CI level: %s'%(perc))

        return Z*stdev/math.sqrt(n)

    def _calc(self, perc = '0.95'):
        self.avg = []
        self.std = []
        self.conf_itv = []
        L=0
        for apth in self.prog_data:
            if len(apth) > L:
                L = len(apth)
        print('\t> calc %s-CI with %d experiments for %s with %s ...'%(perc,len(self.prog_data), self.func, self.alg))
        for i in range(L):
            xs =[]
            x_sum = 0.0
            x_n =0
            for apth in self.prog_data:
                if len(apth) > i:
                    x_n += 1
                    x_sum += apth[i]
                    xs.append(apth[i])
            if x_n < 1:
                self.avg.append(0.0)
                self.std.append(0.0)
                self.conf_itv.append(0.0)
            else:
                x_avg = x_sum/x_n;
                self.avg.append(x_avg)
                vr = 0.0
                for x in xs:
                    vr += (x-x_avg)**2
                xstd = 0.0
                ci = 0.0
                if x_n > 1:
                    xstd = math.sqrt(vr/(x_n-1))

                
                ci = self._calc_ci(perc, x_avg, xstd, x_n)

                self.std.append(xstd)
                self.conf_itv.append(ci)

    def draw_curve(self, ax, linetype, linecolor):
        up = [self.avg[i]+self.conf_itv[i] for i in range(len(self.avg))]
        down = [self.avg[i]-self.conf_itv[i] for i in range(len(self.avg))]
        iters = [i for i in range(len(self.avg))]
        ax.plot(iters, self.avg, color=linecolor, label=self.alg)
        ax.fill_between(iters, down, up, color=linecolor, alpha=0.1)
        

class FuncData:
    def __init__(self, funcname):
        self.func = funcname
        self.algdata = {}
        self.colors = {}

    def append(self, algo_data):
        algo_data.func = self.func
        self.algdata[algo_data.alg] = algo_data

    def set_color(self, alg, color):
        self.colors[alg] = color

    def draw(self, figfn, alg_seq=None, ybottom=None, ytop=None):
        fig,ax = plt.subplots(1,1,figsize=(8,2))
        if ybottom is not None:
            ax.set_ylim(bottom=ybottom)
        if ytop is not None:
            ax.set_ylim(top=ytop)

        if alg_seq is None or len(alg_seq) < 1 :
            alg_seq = self.algdata.keys()
        for alg in alg_seq:
            acolor = self.colors[alg]
            linetype = '--'
            ad = self.algdata[alg]
            ad.draw_curve(ax,linetype, acolor)
        #ax.legend(loc = 'lower center', \
        #         ncol=3, bbox_to_anchor =(0.02,0.15,1,1),\
        #         bbox_transform = plt.gcf().transFigure)
        ax.legend(loc = 'upper center', ncol=4)
        ax.set_xlabel('Number of evaluations')
        ax.set_ylabel('Value')
        ax.set_title(self.func)
        fig.savefig(figfn, dpi=600)

    def sub_draw(self, ax, alg_seq=None, ybottom=None, ytop=None, title=None, xlabel=True):
        if ybottom is not None:
            ax.set_ylim(bottom=ybottom)
        if ytop is not None:
            ax.set_ylim(top=ytop)
        if title is None:
            title = self.func
        if alg_seq is None or len(alg_seq) < 1 :
            alg_seq = self.algdata.keys()
        for alg in alg_seq:
            acolor = self.colors[alg]
            linetype = '--'
            ad = self.algdata[alg]
            ad.draw_curve(ax,linetype, acolor)
        ax.set_ylabel('Value')  
        if xlabel:
            ax.set_xlabel('Number of evaluations')
        ax.set_title(title, x=0.5, y=0.825)  



class DataFiles:
    def __init__(self, directory, func_token, alg_token, alg_rename=None):
    	self.directory = directory
    	self.func_tk = func_token
    	self.alg_tk = alg_token
    	if alg_rename is None:
    		self.alg_show_name = alg_token
    	else:
    		self.alg_show_name = alg_rename
    
    def get_data(self):
    	fns = get_progress_csv_files(self.directory, self.func_tk, self.alg_tk)
    	d = AlgoData(self.alg_show_name, self.func_tk, fns)
    	return d   	


def draw(fname, ax=None):
    global evs
    if fname not in ['cheetah', 'lunar','additive36', 'additive56',\
                          'levy200', 'levy200-large','levy200-large-2','ackley200', 'ackley', 'levy', 'rastrigin',\
                           'hartmann','push', 'rover', 'ackley30', 'levy30', 'rastrigin30']:
        raise ValueError('Unknown function [%s]'%fname)
    data_base =''
    functoken = ''
    dim = 10
    rdir = root_dir()
    ybottom = -10.0
    ytop = 10.0
    ci_perc = '0.95'
    if fname == 'ackley':
        data_base = os.path.join(rdir,'experiments/low-dim-exps/ackley-10-biased')
        functoken = 'Ackley'
        ybottom = 0.0
        ytop = 14.0
    elif fname == 'levy':
        data_base = os.path.join(rdir,'experiments/low-dim-exps/levy-10-biased')
        functoken = 'Levy'
        ybottom = 0.0
        ytop = 40.0
    elif fname == 'rastrigin':
        data_base = os.path.join(rdir,'experiments/low-dim-exps/rastrigin-biased')
        functoken = 'Rastrigin'
        ybottom = 0.0
        ytop = 100.0
    elif fname == 'hartmann':
        data_base = os.path.join(rdir,'experiments/low-dim-exps/hartmann6d')
        functoken = 'Hartmann'
        ybottom = -3.4
        ytop = -1.20
        dim = 6
    elif fname == 'push':
        data_base = os.path.join(rdir, 'experiments/mid-dim-exps/push14d')
        functoken = 'Robot-Push(D14)'
        ybottom = 0.0
        ytop = 13.0
        dim = 14
        evs = 10000
    elif fname == 'rover':
        data_base = os.path.join(rdir, 'experiments/mid-dim-exps/rover60d')
        functoken = 'Rover(D60)'
        ybottom = -4.0
        ytop = 7.0
        dim = 60
        evs = 10000
    elif fname == 'ackley30':
        data_base = os.path.join(rdir,'experiments/mid-dim-exps/ackley-30-biased')
        functoken = 'Ackley'
        ybottom = 0.0
        ytop = 14.0
        dim=30
    elif fname == 'levy30':
        data_base = os.path.join(rdir,'experiments/mid-dim-exps/levy-30-biased')
        functoken = 'Levy'
        ybottom = 0.0
        ytop = 85.0
        dim=30
    elif fname == 'rastrigin30':
        data_base = os.path.join(rdir,'experiments/mid-dim-exps/rastrigin-30-biased')
        functoken = 'Rastrigin'
        ybottom = 0.0
        ytop = 350.0
        dim=30
    elif fname == 'levy200':
        data_base = os.path.join(rdir,'experiments/high-dim-exps/levy-200-biased')
        dim = 200
        functoken = 'Levy'
        ybottom = 0.0
        ytop = 1200.0
        evs = 10000
        ci_perc = '0.995'
    elif fname == 'levy200-large':
        data_base = os.path.join(rdir,'experiments/high-dim-exps/levy-200-biased')
        dim = 200
        functoken = 'Levy'
        ybottom = 0.0
        ytop = 80.0
        evs = 2000
        ci_perc = '0.995'
    elif fname == 'levy200-large-2':
        data_base = os.path.join(rdir,'experiments/high-dim-exps/levy-200-biased')
        dim = 200
        functoken = 'Levy'
        ybottom = 400.0
        ytop = 600.0
        evs = 1000
        ci_perc = '0.995'
    elif fname == 'ackley200':
        data_base = os.path.join(rdir,'experiments/high-dim-exps/ackley-200-biased')
        dim = 200
        functoken = 'Ackley'
        ybottom = 0.0
        ytop = 14.0
        evs = 10000
        ci_perc = '0.995'
    elif fname == 'additive36':
        data_base=os.path.join(rdir, 'experiments/additive-exps/additive-36d')
        dim = 36
        functoken = 'Additive'
        ybottom = 0.0
        ytop = 175.0
        evs = 5000
    elif fname == 'additive56':
        data_base=os.path.join(rdir, 'experiments/additive-exps/additive-56d')
        dim = 56
        functoken = 'Additive'
        ybottom = 0.0
        ytop = 4.0e4
        evs = 10000
    elif fname == 'lunar':
        data_base = os.path.join(rdir, 'experiments/mid-dim-exps/lunar12d')
        dim = 12
        functoken = 'Lunar'
        ybottom = 0.0
        ytop = 350.0
        evs = 1500
    elif fname == 'cheetah':
        data_base = os.path.join(rdir,'experiments/mujoco-exps/test/exp0524')
        dim = 102
        functoken = 'CheetahV2'
        dim = 102
        ytop = 4500.0
        ybottom = -100.0
        evs = 10000
    else:
        raise ValueError('Should not have arrived here!!!')

    bads_ftk = functoken
    if fname == 'push':
        bads_ftk = 'push14d'
    elif fname == 'rover':
        bads_ftk = 'Rover(D60)'
        evs = 20000

    fdata = None
    if fname == 'push':
        fdata = FuncData('14D Robot pushing')
    elif fname == 'rover':
        fdata = FuncData('60D Rover trajectory planning')
    elif fname == 'additive36':
        fdata = FuncData('36D Additive function')
    elif fname == 'additive56':
        fdata = FuncData('56D Additive function')
    elif fname == 'lunar':
        fdata = FuncData('12D Lunar landing')
    elif fname == 'cheetah':
        fdata = FuncData('102D half-cheetah control')
    else:
        fdata = FuncData('%dD %s function'%(dim,functoken))

    rvs_sign = False
    if fname == 'push':
        rvs_sign = True
    elif fname == 'rover':
        rvs_sign = False #has been processed in data

    if dim < 50:
        atpe_csvs = get_progress_csv_files(os.path.join(data_base, 'atpe'), functoken, 'ATPE')
        atpe_ad = AlgoData('ATPE', functoken, atpe_csvs, perc=ci_perc)
        fdata.append(atpe_ad)
        fdata.set_color(atpe_ad.alg, 'red')

    if dim < 60:
        bads_csvs = get_progress_csv_files(os.path.join(data_base, 'bads'), bads_ftk, 'BADS')
        bads_ad = AlgoData('BADS', functoken, bads_csvs,  perc=ci_perc, reverse_sign=rvs_sign)
        fdata.append(bads_ad)
        fdata.set_color(bads_ad.alg, 'green')


    cmaes_csvs = get_progress_csv_files(os.path.join(data_base, 'cmaes'), functoken, 'CMAES')

    cobbo_base = os.path.join(data_base, 'cobbo')
    cobbo_csvs = get_progress_csv_files(cobbo_base, functoken, 'CobBO(new)') ##

    rembo_csvs = None
    if dim < 13:
        rembo_csvs = get_progress_csv_files(os.path.join(data_base, 'rembo'), functoken, 'REMBO')
        rembo_ad = AlgoData('REMBO', functoken, rembo_csvs, perc=ci_perc)
    
    tpe_csvs = get_progress_csv_files(os.path.join(data_base, 'tpe'), functoken, 'TPE')
    turbo_csvs = get_progress_csv_files(os.path.join(data_base, 'turbo'), functoken, 'TuRBO')


    cmaes_ad = AlgoData('CMAES', functoken, cmaes_csvs, perc=ci_perc)
    cobbo_ad = AlgoData('CobBO', functoken, cobbo_csvs, perc=ci_perc)
    
    #
    tpe_ad = AlgoData('TPE', functoken, tpe_csvs, perc=ci_perc)
    turbo_ad = AlgoData('TuRBO', functoken, turbo_csvs, perc=ci_perc)

    fdata.append(cmaes_ad)
    fdata.append(cobbo_ad)    
    
    fdata.append(tpe_ad)
    fdata.append(turbo_ad)

    
    
    fdata.set_color(cmaes_ad.alg, 'fuchsia')
    fdata.set_color(cobbo_ad.alg, 'black')
    fdata.set_color(tpe_ad.alg, 'orange')
    fdata.set_color(turbo_ad.alg, 'blue')

    if ax is None:
        figname = '%s-%dd.png'%(fname,dim)
        fdata.draw(figname,ybottom=ybottom, ytop=ytop)
    else:
        fdata.sub_draw(ax,ybottom=ybottom, ytop=ytop, xlabel=True)



def draw_old_new(func, ax=None):
    if func not in ['push', 'rover']:
        raise ValueError('Invalid function: [%s] . Must be [push] or [rover].')
    topdir = script_dir()
    newdir = os.path.join(topdir,'new')
    olddir = os.path.join(topdir,'old')

    dim = 10
    ytop = 100.0
    ybottom = -100.0

    functoken = '~~'
    if func == 'push':
        functoken = 'Robot-Push(D14)'
        dim = 14
        ytop = 11.0
        ybottom = 4.0
    elif func == 'rover':
        functoken = 'Rover(D60)'
        dim = 60
        ytop = 4.2
        ybottom = -1.0


    new_csvs = get_progress_csv_files(newdir, functoken, 'CobBO(new)')
    old_csvs = get_progress_csv_files(olddir, functoken, 'CobBO(old)')

    new_ad = AlgoData('CobBO(new)', functoken, new_csvs)
    old_ad = AlgoData('CobBO(old)', functoken, old_csvs)

    fdata = None
    if func == 'push':
        fdata = FuncData('14D Robot pushing')
    elif func == 'rover':
        fdata = FuncData('60D Rover trajectory planning')


    fdata.append(new_ad)
    fdata.append(old_ad)

    fdata.set_color(new_ad.alg, 'black')
    fdata.set_color(old_ad.alg, 'orange')

    if ax is None:
        figname = '%s-%dd.png'%(func,dim)
        fdata.draw(figname,ybottom=ybottom, ytop=ytop)
    else:
        fdata.sub_draw(ax,ybottom=ybottom, ytop=ytop)



    

if __name__ == '__main__xxx':
    fig,axs = plt.subplots(1,2, figsize=(8,3.0)) 
    draw_old_new('push',axs[0])
    draw_old_new('rover',axs[1])
    handles, labels = axs[0].get_legend_handles_labels()
    #axs[0].legend(handles, labels, loc='upper left', 
    #         bbox_to_anchor=(-0.1, -0.4),fancybox=False, shadow=False, ncol=2)
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=2, bbox_to_anchor =(0.02,-0.06,1,1),\
                 bbox_transform = plt.gcf().transFigure)
    plt.tight_layout()
    fig.savefig('cobbo-old-new.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')


def draw_push_lander():
    print('sdir=',script_dir())
    print('rdir=',root_dir())
    fs = ['push', 'lunar']
    fig,axs = plt.subplots(1,2, figsize=(7,2.8))
    draw(fs[0],axs[0])
    draw(fs[1],axs[1])
    handles, labels = axs[0].get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=6, bbox_to_anchor =(0.02,-0.08,1,1),\
                 bbox_transform = plt.gcf().transFigure)
    plt.tight_layout()
    fig.savefig('lunar-push.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')

def draw_rover():
    print('sdir=',script_dir())
    print('rdir=',root_dir())
    fs = ['rover']
    fig,axs = plt.subplots(1,1, figsize=(4,2.5))
    draw(fs[0],axs)
    handles, labels = axs.get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=1, bbox_to_anchor =(0.6,0.45,1,1),\
                 bbox_transform = plt.gcf().transFigure)
    plt.tight_layout()
    fig.savefig('rover60d.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')


def draw_additives():
    print('sdir=',script_dir())
    print('rdir=',root_dir())
    fs = ['additive36', 'additive56']
    fig,axs = plt.subplots(1,2, figsize=(7,2.5))
    draw(fs[0],axs[0])
    draw(fs[1],axs[1])
    handles, labels = axs[0].get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=6, bbox_to_anchor =(0.02,-0.07,1,1),\
                 bbox_transform = plt.gcf().transFigure)
    plt.tight_layout()
    fig.savefig('additives.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')



def _draw_cheetah(ax=None):
    global evs
    evs = 4000
    func = 'cheetah'
    topdir = script_dir()
    datadir = os.path.join(root_dir(),'experiments/mujoco-exps/test/exp0524')
    
    dim = 102
    ytop = 5000.0
    ybottom = -100.0

    functoken = 'CheetahV2'


    tpe_csvs = get_progress_csv_files(datadir, functoken, 'TPE')
    turbo_csvs = get_progress_csv_files(datadir, functoken, 'TuRBO')
    cma_csvs = get_progress_csv_files(datadir, functoken, 'CMAES')
    cob_csvs = get_progress_csv_files(datadir, functoken, 'CobBO(new)')

    tpe_ad = AlgoData('TPE', functoken, tpe_csvs)
    turbo_ad = AlgoData('TuRBO', functoken, turbo_csvs)
    cma_ad = AlgoData('CMAES', functoken, cma_csvs)
    cob_ad = AlgoData('CobBO', functoken, cob_csvs)


    fdata = FuncData('102D Half-cheetah control')

    fdata.append(tpe_ad)
    fdata.append(turbo_ad)
    fdata.append(cma_ad)
    fdata.append(cob_ad)

    fdata.set_color(tpe_ad.alg, 'orange')
    fdata.set_color(turbo_ad.alg, 'blue')
    fdata.set_color(cma_ad.alg, 'fuchsia')
    fdata.set_color(cob_ad.alg, 'black')

    if ax is None:
        figname = '%s-%dd.png'%(func,dim)
        fdata.draw(figname,ybottom=ybottom, ytop=ytop)
    else:
        fdata.sub_draw(ax,ybottom=ybottom, ytop=ytop)



def draw_cheetah():
    print('sdir=',script_dir())
    print('rdir=',root_dir())
    fs = ['cheetah']
    fig,axs = plt.subplots(1,1, figsize=(4,2.5))
    _draw_cheetah(axs)
    handles, labels = axs.get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=1, bbox_to_anchor =(0.6,0.45,1,1),\
                 bbox_transform = plt.gcf().transFigure)
    plt.tight_layout()
    fig.savefig('cheetah.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')


def draw_10d(func, ax):
    global evs 
    evs = 500
    xdir = os.path.join(root_dir() , 'experiments/low-dim-exps')
    basedir =None
    ytop = 10.0
    ybottom = -10.0
    functoken = None
    if func == 'ackley10':
        basedir = os.path.join(xdir,'ackley-10-biased')
        ybottom = 0.0
        ytop = 14.0
        functoken = 'Ackley'
    elif func == 'levy10':
        basedir = os.path.join(xdir, 'levy-10-biased')
        ybottom = 0.0
        ytop = 40.0
        functoken = 'Levy'
    elif func == 'rastrigin10':
        basedir = os.path.join(xdir, 'rastrigin-biased')
        ybottom = 0.0
        ytop = 110.0
        functoken = 'Rastrigin'
    atpe_csvs = get_progress_csv_files(basedir+'/atpe',functoken,'ATPE')
    bads_csvs = get_progress_csv_files(basedir+'/bads',functoken, 'BADS')
    cmaes_csvs= get_progress_csv_files(basedir+'/cmaes',functoken,'CMAES')
    rembo_csvs= get_progress_csv_files(basedir+'/rembo',functoken,'REMBO')
    tpe_csvs  = get_progress_csv_files(basedir+'/tpe', functoken, 'TPE')
    turbo_csvs= get_progress_csv_files(basedir+'/turbo',functoken, 'TuRBO')

    cobbo_csvs= get_progress_csv_files(basedir+'/cobbo',functoken,'CobBO')

    atpe_ad = AlgoData('ATPE', functoken, atpe_csvs)
    bads_ad = AlgoData('BADS', functoken, bads_csvs)
    cma_ad  = AlgoData('CMA-ES', functoken, cmaes_csvs)
    rembo_ad= AlgoData('REMBO', functoken, rembo_csvs)
    tpe_ad  = AlgoData('TPE', functoken, tpe_csvs)
    turbo_ad= AlgoData('TuRBO', functoken, turbo_csvs)
    cobbo_ad= AlgoData('CobBO', functoken, cobbo_csvs)

    fdata = FuncData('10D %s function'%functoken)
    fdata.append(atpe_ad)
    fdata.append(bads_ad)
    fdata.append(cma_ad)
    fdata.append(rembo_ad)
    fdata.append(tpe_ad)
    fdata.append(turbo_ad)
    fdata.append(cobbo_ad)

    fdata.set_color(atpe_ad.alg, 'red')
    fdata.set_color(bads_ad.alg, 'green')
    fdata.set_color(cma_ad.alg, 'fuchsia')
    fdata.set_color(rembo_ad.alg, 'purple')
    fdata.set_color(tpe_ad.alg, 'orange')
    fdata.set_color(turbo_ad.alg, 'blue')
    fdata.set_color(cobbo_ad.alg, 'black')

    fdata.sub_draw(ax, ybottom=ybottom, ytop=ytop, xlabel=False)



def draw_30d(func, ax):
    global evs 
    evs = 5000
    xdir = os.path.join(root_dir() , 'experiments/mid-dim-exps')
    basedir =None
    ytop = 10.0
    ybottom = -10.0
    functoken = None
    if func == 'ackley30':
        basedir = os.path.join(xdir,'ackley-30-biased')
        ybottom = 0.0
        ytop = 14.0
        functoken = 'Ackley'
    elif func == 'levy30':
        basedir = os.path.join(xdir, 'levy-30-biased')
        ybottom = 0.0
        ytop = 100.0
        functoken = 'Levy'
    elif func == 'rastrigin30':
        basedir = os.path.join(xdir, 'rastrigin-30-biased')
        ybottom = 0.0
        ytop = 400.0
        functoken = 'Rastrigin'

    print('basedir = ', basedir)
    atpe_csvs = get_progress_csv_files(basedir+'/atpe',functoken,'ATPE')
    bads_csvs = get_progress_csv_files(basedir+'/bads',functoken, 'BADS')
    cmaes_csvs= get_progress_csv_files(basedir+'/cmaes',functoken,'CMAES')
    #rembo_csvs= get_progress_csv_files(basedir+'/rembo',functoken,'REMBO')
    tpe_csvs  = get_progress_csv_files(basedir+'/tpe', functoken, 'TPE')
    turbo_csvs= get_progress_csv_files(basedir+'/turbo',functoken, 'TuRBO')

    cobbo_csvs= get_progress_csv_files(basedir+'/cobbo',functoken,'CobBO(new)')

    atpe_ad = AlgoData('ATPE', functoken, atpe_csvs)
    bads_ad = AlgoData('BADS', functoken, bads_csvs)
    cma_ad  = AlgoData('CMA-ES', functoken, cmaes_csvs)
    #rembo_ad= AlgoData('REMBO', functoken, rembo_csvs)
    tpe_ad  = AlgoData('TPE', functoken, tpe_csvs)
    turbo_ad= AlgoData('TuRBO', functoken, turbo_csvs)
    cobbo_ad= AlgoData('CobBO', functoken, cobbo_csvs)

    fdata = FuncData('30D %s function'%functoken)
    fdata.append(atpe_ad)
    fdata.append(bads_ad)
    fdata.append(cma_ad)
    #fdata.append(rembo_ad)
    fdata.append(tpe_ad)
    fdata.append(turbo_ad)
    fdata.append(cobbo_ad)

    fdata.set_color(atpe_ad.alg, 'red')
    fdata.set_color(bads_ad.alg, 'green')
    fdata.set_color(cma_ad.alg, 'fuchsia')
    #fdata.set_color(rembo_ad.alg, 'purple')
    fdata.set_color(tpe_ad.alg, 'orange')
    fdata.set_color(turbo_ad.alg, 'blue')
    fdata.set_color(cobbo_ad.alg, 'black')

    fdata.sub_draw(ax, ybottom=ybottom, ytop=ytop, xlabel=True)




def draw_100d(func, ax):
    global evs 
    evs = 5000
    xdir = os.path.join(root_dir() , 'raw_data/Synthetic')
    basedir =None
    ytop = 10.0
    ybottom = -10.0
    functoken = None
    if func == 'ackley100':
        basedir = xdir #os.path.join(xdir,'ackley-30-biased')
        ybottom = 0.0
        ytop = 14.0
        functoken = 'Ackley'
    elif func == 'levy100':
        basedir = xdir #os.path.join(xdir, 'levy-30-biased')
        ybottom = 0.0
        ytop = 600.0
        functoken = 'Levy'
    elif func == 'rastrigin100':
        basedir = xdir #os.path.join(xdir, 'rastrigin-30-biased')
        ybottom = 0.0
        ytop = 3500.0
        functoken = 'Rastrigin'

    print('basedir = ', basedir)
    atpe_csvs = get_progress_csv_files(basedir+'/atpe',functoken,'ATPE')
    bads_csvs = get_progress_csv_files(basedir+'/bads',functoken, 'BADS')
    cmaes_csvs= get_progress_csv_files(basedir+'/cmaes',functoken,'CMAES')
    #rembo_csvs= get_progress_csv_files(basedir+'/rembo',functoken,'REMBO')
    tpe_csvs  = get_progress_csv_files(basedir+'/tpe', functoken, 'TPE')
    turbo_csvs= get_progress_csv_files(basedir+'/turbo',functoken, 'TuRBO')

    cobbo_csvs= get_progress_csv_files(basedir+'/cobbo',functoken,'CobBO(new)')

    atpe_ad = AlgoData('ATPE', functoken, atpe_csvs)
    bads_ad = AlgoData('BADS', functoken, bads_csvs)
    cma_ad  = AlgoData('CMA-ES', functoken, cmaes_csvs)
    #rembo_ad= AlgoData('REMBO', functoken, rembo_csvs)
    tpe_ad  = AlgoData('TPE', functoken, tpe_csvs)
    turbo_ad= AlgoData('TuRBO', functoken, turbo_csvs)
    cobbo_ad= AlgoData('CobBO', functoken, cobbo_csvs)

    fdata = FuncData('100D %s function'%functoken)
    fdata.append(atpe_ad)
    fdata.append(bads_ad)
    fdata.append(cma_ad)
    #fdata.append(rembo_ad)
    fdata.append(tpe_ad)
    fdata.append(turbo_ad)
    fdata.append(cobbo_ad)

    fdata.set_color(atpe_ad.alg, 'red')
    fdata.set_color(bads_ad.alg, 'green')
    fdata.set_color(cma_ad.alg, 'fuchsia')
    #fdata.set_color(rembo_ad.alg, 'purple')
    fdata.set_color(tpe_ad.alg, 'orange')
    fdata.set_color(turbo_ad.alg, 'blue')
    fdata.set_color(cobbo_ad.alg, 'black')

    fdata.sub_draw(ax, ybottom=ybottom, ytop=ytop, xlabel=True)



def draw_synthetic_functions():
    fig,axs = plt.subplots(2,3, figsize=(10,4))
    
    #ackley10
    #levy10
    #rastrigin10
    draw_10d('ackley10',    axs[0,0])
    draw_10d('levy10',      axs[0,1])
    draw_10d('rastrigin10', axs[0,2])

    draw_30d('ackley30',   axs[1,0])
    draw_30d('levy30',     axs[1,1])
    draw_30d('rastrigin30',axs[1,2])

    handles, labels = axs[0,1].get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=7, bbox_to_anchor =(0.02,-0.06,1,1),\
                 bbox_transform = plt.gcf().transFigure)

    plt.tight_layout()
    #plt.figure(figsize=(1,0.3))
    fig.savefig('synthetic.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')
    

def draw_fig_3():
    fig,axs = plt.subplots(2,3, figsize=(10,4))
    
    #ackley10
    #levy10
    #rastrigin10
    draw('lunar',           axs[0,0])
    draw('additive56',      axs[0,1])
    draw('levy100',         axs[0,2])
    draw('push',            axs[1,0])
    draw('rover',           axs[1,1])
    draw('rastrigin100',    axs[1,2])

    #draw_30d('ackley30',   axs[1,0])
    #draw_30d('levy30',     axs[1,1])
    #draw_30d('rastrigin30',axs[1,2])

    handles, labels = axs[0,1].get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=7, bbox_to_anchor =(0.02,-0.06,1,1),\
                 bbox_transform = plt.gcf().transFigure)

    plt.tight_layout()
    #plt.figure(figsize=(1,0.3))
    fig.savefig('fig3.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')
  

def draw_controls_1():
    fig,axs = plt.subplots(1,2, figsize=(8,2.5))
    fig.subplots_adjust(hspace=0.4, wspace=0.4)
    draw('lunar',axs[0])
    draw('push', axs[1])

    handles, labels = axs[0].get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=7, bbox_to_anchor =(0.02,-0.09,1,1),\
                 bbox_transform = plt.gcf().transFigure)

    plt.tight_layout()
    plt.figure(figsize=(1,0.3))
    fig.savefig('lunar-robot.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')
   

def draw_controls_2():
    fig,axs = plt.subplots(1,3, figsize=(11,2.7))
    
    draw('additive36', axs[0])
    draw('additive56', axs[1])
    
    draw('rover', axs[2])
    #_draw_cheetah(axs[3])

    handles, labels = axs[0].get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=7, bbox_to_anchor =(0.02,-0.07,1,1),\
                 bbox_transform = plt.gcf().transFigure)

    plt.tight_layout()
    #plt.figure(figsize=(1,0.25))
    fig.savefig('highDims.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')
   
def draw_200d():
    print('sdir=',script_dir())
    print('rdir=',root_dir())
    fs = ['levy200', 'ackley200']
    fig,axs = plt.subplots(1,3, figsize=(12,3))
    #fig,axs = plt.subplots(1,3, figsize=(60,15))
    fig.subplots_adjust(hspace=0.4, wspace=0.4)
    draw(fs[0],axs[0])
    draw(fs[1],axs[1])
    _draw_cheetah(axs[2])
    
    handles, labels = axs[0].get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=7, bbox_to_anchor =(0.02,-0.07,1,1),\
                 bbox_transform = plt.gcf().transFigure)
    plt.tight_layout()
    fig.savefig('levy200d.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')


def draw_200_large():
    print('sdir=',script_dir())
    print('rdir=',root_dir())
    fs = ['levy200', 'ackley200']
    #fig,axs = plt.subplots(1,3, figsize=(60,15))
    fig,axs = plt.subplots(1,2, figsize=(10,3.5))
    fig.subplots_adjust(hspace=0.4, wspace=0.4)
    #draw(fs[0],axs[0])
    #draw(fs[1],axs[1])
    #_draw_cheetah(axs[2])
    draw('levy200-large-2',axs[0])
    draw('levy200-large',axs[1])
    
    
    handles, labels = axs[0].get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=7, bbox_to_anchor =(0.02,-0.07,1,1),\
                 bbox_transform = plt.gcf().transFigure)
    plt.tight_layout()
    fig.savefig('200d-large.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')


def draw_200_large_full():
    print('sdir=',script_dir())
    print('rdir=',root_dir())
    fs = ['levy200', 'ackley200']
    #fig,axs = plt.subplots(1,3, figsize=(60,15))
    fig,axs = plt.subplots(1,1, figsize=(12,8))
    fig.subplots_adjust(hspace=0.4, wspace=0.4)
    #draw(fs[0],axs[0])
    #draw(fs[1],axs[1])
    #_draw_cheetah(axs[2])
    draw('levy200',axs)
    
    handles, labels = axs.get_legend_handles_labels()
    lgd = fig.legend(handles, labels, loc = 'lower center', \
                 ncol=7, bbox_to_anchor =(0.02,-0.07,1,1),\
                 bbox_transform = plt.gcf().transFigure)
    plt.tight_layout()
    fig.savefig('200d-large-full.png', format='png', dpi=600, bbox_extra_artists=(lgd,), bbox_inches='tight')

if __name__ == '__main__':
    draw_fig_3()

if __name__ == 'xxxx__main__':
    draw_synthetic_functions()
    draw_controls_1()
    draw_controls_2()
    draw_200d()
    draw_200_large()
    draw_200_large_full()


