import numpy as np
import random
from datetime import datetime
import torch
import logging
import logging.handlers
import os

from .file_io import mkdir


def set_seeds(rng_seed : int, use_cuda : bool) -> None:
    np.random.seed(rng_seed)
    torch.manual_seed(rng_seed)
    random.seed(rng_seed)
    if use_cuda:
        torch.cuda.manual_seed(rng_seed)
        torch.backends.cudnn.deterministic = True
        torch.backends.cudnn.benchmark = False


def setup_logging(output_dir : str = "", level=logging.INFO, stream : bool = True) -> None:
    formatter = logging.Formatter(
        fmt='%(asctime)s,%(msecs)03d %(levelname)s '
            '[%(filename)s:%(lineno)s - %(funcName)s] - %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S')

    logger = logging.getLogger()
    logger.setLevel(level)
    if stream or not output_dir:
        handler = logging.StreamHandler()
        handler.setFormatter(formatter)
        handler.setLevel(level)
        logger.addHandler(handler)
    if output_dir:
        mkdir(output_dir)
        date_time = datetime.now().strftime("%Y-%m-%d_%H:%M")
        handler = logging.handlers.RotatingFileHandler(
            os.path.join(output_dir, "log.{}".format(date_time)),
            encoding="utf-8",
            maxBytes=100 * 1024 * 1024,  # 100M
            backupCount=100
        )
        handler.setFormatter(formatter)
        handler.setLevel(level)
        logger.addHandler(handler)

    logger.info("Setting up logger complete.")


def get_logname(logger):
    if len(logger.root.handlers) == 1:
        return None
    else:
        return logger.root.handlers[1].baseFilename
