# Copyright (c) 2015-present, Facebook, Inc.
# All rights reserved.


from functools import partial
from typing import Optional

import torch
import torch.nn as nn
import torch.nn.functional as F
from timm.models.efficientnet_blocks import SqueezeExcite
from timm.models.vision_transformer import Mlp, PatchEmbed , _cfg
from timm.models.layers import DropPath, to_2tuple, trunc_normal_
from timm.models.registry import register_model

import numpy as np

from torchvision import transforms
#import torch.nn.functional as F
import matplotlib.pyplot as plt
from matplotlib.gridspec import GridSpec

from matplotlib.patches import Rectangle

from scipy.interpolate import griddata
import time

from skimage.measure import block_reduce
import torchvision.models as torch_models

from xcit import PositionalEncodingFourier


inv_transform = transforms.Compose([
    transforms.Normalize(mean=[-0.485/0.229, -0.456/0.224, -0.406/0.225], std=[1.0/0.229, 1.0/0.224, 1.0/0.225]),
    transforms.ToPILImage(mode='RGB'),
])


__all__ = ['S60', 'S120', 'B60', 'B120', 'L60', 'L120', 'S60_multi']

import json
class_names = json.load(open("imagenet_class_index.json"))


'''
 1 - Fovea 1x1, Retain 1x1
 2 - Fovea 3x3, Retain 1x1
 3 - Fovea 3x3, Retain 3x3
'''
flag_FR = 3 #1 
#flag_PreLayers = 6 #6
flag_fullres = True
flag_uni = False


backbone_size = 'Medium' #'Large' #'Medium' # Small, Large

class Attention(nn.Module):
    # taken from https://github.com/rwightman/pytorch-image-models/blob/master/timm/models/vision_transformer.py
    def __init__(self, dim, num_heads=8, qkv_bias=False, qk_scale=None, attn_drop=0., proj_drop=0.):
        super().__init__()
        self.num_heads = num_heads
        head_dim = dim // num_heads
        self.scale = qk_scale or head_dim ** -0.5

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

    def forward(self, x):
        B, N, C = x.shape
        qkv = self.qkv(x).reshape(B, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv[0], qkv[1], qkv[2]
        
        q = q * self.scale

        attn = (q @ k.transpose(-2, -1))
        attn = attn.softmax(dim=-1)
        attn = self.attn_drop(attn)

        x = (attn @ v).transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x
    
class Block(nn.Module):
    # taken from https://github.com/rwightman/pytorch-image-models/blob/master/timm/models/vision_transformer.py
    def __init__(self, dim, num_heads, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., act_layer=nn.GELU, norm_layer=nn.LayerNorm,Attention_block = Attention,Mlp_block=Mlp
                 ,init_values=1e-4):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = Attention_block(
            dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)
        # NOTE: drop path for stochastic depth, we shall see if this is better than dropout here
        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp_block(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

    def forward(self, x):
        x = x + self.drop_path(self.attn(self.norm1(x)))
        x = x + self.drop_path(self.mlp(self.norm2(x)))
        return x 
    
class Self_Attention_Layer_scale_init_Block(nn.Module):
    # taken from https://github.com/rwightman/pytorch-image-models/blob/master/timm/models/vision_transformer.py
    # with slight modifications
    def __init__(self, dim, num_heads, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., act_layer=nn.GELU, norm_layer=nn.LayerNorm,Attention_block = Attention,Mlp_block=Mlp
                 ,init_values=1e-4):
        super().__init__()
        #print('Entered Self_Attention_Layer_scale_init_Block')
        self.norm1 = norm_layer(dim)
        self.attn = Attention_block(
            dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)
        # NOTE: drop path for stochastic depth, we shall see if this is better than dropout here
        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp_block(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)
        self.gamma_1 = nn.Parameter(init_values * torch.ones((dim)),requires_grad=True)
        self.gamma_2 = nn.Parameter(init_values * torch.ones((dim)),requires_grad=True)

    def forward(self, x):
        #print('Self_Attention_Layer_scale_init_Block: x.shape: {}, norm1: {}'.format(x.shape, self.norm1))
        x = x + self.drop_path(self.gamma_1 * self.attn(self.norm1(x)))
        x = x + self.drop_path(self.gamma_2 * self.mlp(self.norm2(x)))
        return x


class Mlp(nn.Module):
    def __init__(
        self,
        in_features: int,
        hidden_features: Optional[int] = None,
        out_features: Optional[int] = None,
        act_layer: nn.Module = nn.GELU,
        drop: float = 0.0,
    ):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        self.fc1 = nn.Linear(in_features, hidden_features)
        self.act = act_layer()
        self.fc2 = nn.Linear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


class Learned_Aggregation_Layer(nn.Module):
    def __init__(
        self,
        dim: int,
        num_heads: int = 1,
        qkv_bias: bool = False,
        qk_scale: Optional[float] = None,
        attn_drop: float = 0.0,
        proj_drop: float = 0.0,
    ):
        super().__init__()
        self.num_heads = num_heads
        head_dim: int = dim // num_heads
        # NOTE scale factor was wrong in my original version, can set manually to be compat with prev weights
        self.scale = qk_scale or head_dim**-0.5

        self.q = nn.Linear(dim, dim, bias=qkv_bias)
        self.k = nn.Linear(dim, dim, bias=qkv_bias)
        self.v = nn.Linear(dim, dim, bias=qkv_bias)
        self.id = nn.Identity()
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

    def forward(self, x: torch.Tensor):# -> torch.Tensor:
        B, N, C = x.shape
        #print('Learned_Aggregation_Layer #1: x.shape: {}'.format(x.shape))
        q = self.q(x[:, 0]).unsqueeze(1).reshape(B, 1, self.num_heads, C // self.num_heads).permute(0, 2, 1, 3)
        k = self.k(x).reshape(B, N, self.num_heads, C // self.num_heads).permute(0, 2, 1, 3)

        q = q * self.scale
        v = self.v(x).reshape(B, N, self.num_heads, C // self.num_heads).permute(0, 2, 1, 3)

        attn = q @ k.transpose(-2, -1)
        #print('Learned_Aggregation_Layer #2: attn.shape: {}'.format(attn.shape))
        attn = self.id(attn)
        attn = attn.softmax(dim=-1)
        attn = self.attn_drop(attn)
        #print('AJ: Learned_Aggregation_Layer #3: attn.shape: {}'.format(attn.shape))

        x_cls = (attn @ v).transpose(1, 2).reshape(B, 1, C)
        x_cls = self.proj(x_cls)
        x_cls = self.proj_drop(x_cls)

        #print('Learned_Aggregation_Layer #4: x_cls.shape: {}'.format(x_cls.shape))
        return x_cls, attn


class Learned_Aggregation_Layer_multi(nn.Module):
    def __init__(
        self,
        dim: int,
        num_heads: int = 8,
        qkv_bias: bool = False,
        qk_scale: Optional[float] = None,
        attn_drop: float = 0.0,
        proj_drop: float = 0.0,
        num_classes: int = 1000,
    ):
        super().__init__()
        self.num_heads = num_heads
        head_dim: int = dim // num_heads
        self.scale = qk_scale or head_dim**-0.5

        self.q = nn.Linear(dim, dim, bias=qkv_bias)
        self.k = nn.Linear(dim, dim, bias=qkv_bias)
        self.v = nn.Linear(dim, dim, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)
        self.num_classes = num_classes

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        B, N, C = x.shape
        q = (
            self.q(x[:, : self.num_classes])
            .reshape(B, self.num_classes, self.num_heads, C // self.num_heads)
            .permute(0, 2, 1, 3)
        )
        k = (
            self.k(x[:, self.num_classes:])
            .reshape(B, N - self.num_classes, self.num_heads, C // self.num_heads)
            .permute(0, 2, 1, 3)
        )

        q = q * self.scale
        v = (
            self.v(x[:, self.num_classes:])
            .reshape(B, N - self.num_classes, self.num_heads, C // self.num_heads)
            .permute(0, 2, 1, 3)
        )

        attn = q @ k.transpose(-2, -1)
        attn = attn.softmax(dim=-1)
        attn = self.attn_drop(attn)

        x_cls = (attn @ v).transpose(1, 2).reshape(B, self.num_classes, C)
        x_cls = self.proj(x_cls)
        x_cls = self.proj_drop(x_cls)

        return x_cls, attn


class Layer_scale_init_Block_only_token(nn.Module):
    def __init__(
        self,
        dim: int,
        num_heads: int,
        mlp_ratio: float = 4.0,
        qkv_bias: bool = False,
        qk_scale: Optional[float] = None,
        drop: float = 0.0,
        attn_drop: float = 0.0,
        drop_path: float = 0.0,
        act_layer: nn.Module = nn.GELU,
        norm_layer=nn.LayerNorm,
        Attention_block=Learned_Aggregation_Layer,
        Mlp_block=Mlp,
        init_values: float = 1e-4,
    ):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = Attention_block(
            dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop
        )
        # NOTE: drop path for stochastic depth, we shall see if this is better than dropout here
        self.drop_path = DropPath(drop_path) if drop_path > 0.0 else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp_block(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)
        self.gamma_1 = nn.Parameter(init_values * torch.ones((dim)), requires_grad=True)
        self.gamma_2 = nn.Parameter(init_values * torch.ones((dim)), requires_grad=True)

    def forward(self, x: torch.Tensor, x_cls: torch.Tensor): # -> torch.Tensor:
        u = torch.cat((x_cls, x), dim=1)
        #print('Layer_scale_init_Block_only_token: x.shape: {}, x_cls.shape: {}, u.shape: {}'.format(x.shape, x_cls.shape, u.shape))
        u_SA, attn = self.attn(self.norm1(u))
        x_cls = x_cls + self.drop_path(self.gamma_1 * u_SA)
        x_cls = x_cls + self.drop_path(self.gamma_2 * self.mlp(self.norm2(x_cls)))
        return x_cls, attn


class Conv_blocks_se(nn.Module):
    def __init__(self, dim: int):
        super().__init__()

        self.qkv_pos = nn.Sequential(
            nn.Conv2d(dim, dim, kernel_size=1),
            nn.GELU(),
            nn.Conv2d(dim, dim, groups=dim, kernel_size=3, padding=1, stride=1, bias=True),
            nn.GELU(),
            SqueezeExcite(dim, rd_ratio=0.25),
            nn.Conv2d(dim, dim, kernel_size=1),
        )

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        B, N, C = x.shape
        H = W = int(N ** 0.5)
        x = x.transpose(-1, -2)
        x = x.reshape(B, C, H, W)
        x = self.qkv_pos(x)
        x = x.reshape(B, C, N)
        x = x.transpose(-1, -2)
        return x


class Layer_scale_init_Block(nn.Module):
    def __init__(
        self,
        dim: int,
        drop_path: float = 0.0,
        act_layer: nn.Module = nn.GELU,
        norm_layer=nn.LayerNorm,
        Attention_block=None,
        init_values: float = 1e-4,
    ):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = Attention_block(dim)
        self.drop_path = DropPath(drop_path) if drop_path > 0.0 else nn.Identity()
        self.gamma_1 = nn.Parameter(init_values * torch.ones((dim)), requires_grad=True)

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        return x + self.drop_path(self.gamma_1 * self.attn(self.norm1(x)))


def conv3x3(in_planes: int, out_planes: int, stride: int = 1) -> nn.Sequential:
    """3x3 convolution with padding"""
    return nn.Sequential(
        nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=False),
    )


class ConvStem(nn.Module):
    """Image to Patch Embedding"""

    def __init__(self, img_size: int = 224, patch_size: int = 16, in_chans: int = 3, embed_dim: int = 768):
        super().__init__()
        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)
        num_patches = (img_size[1] // patch_size[1]) * (img_size[0] // patch_size[0])
        self.img_size = img_size
        self.patch_size = patch_size
        self.num_patches = num_patches

        self.proj = nn.Sequential(
            conv3x3(in_chans, embed_dim // 8, 2),
            nn.GELU(),
            conv3x3(embed_dim // 8, embed_dim // 4, 2),
            nn.GELU(),
            conv3x3(embed_dim // 4, embed_dim // 2, 2),
            nn.GELU(),
            conv3x3(embed_dim // 2, embed_dim, 2),
        )

    def forward(self, x: torch.Tensor, padding_size: Optional[int] = None) -> torch.Tensor:
        B, C, H, W = x.shape
        x = self.proj(x).flatten(2).transpose(1, 2)
        return x


class PatchConvnet(nn.Module):
    def __init__(
        self,
        img_size: int = 224,
        patch_size: int = 16,
        in_chans: int = 3,
        num_classes: int = 1000,
        embed_dim: int = 768,
        depth: int = 12,
        num_heads: int = 1,
        qkv_bias: bool = False,
        qk_scale: Optional[float] = None,
        drop_rate: float = 0.0,
        attn_drop_rate: float = 0.0,
        drop_path_rate: float = 0.0,
        hybrid_backbone: Optional = None,
        norm_layer=nn.LayerNorm,
        global_pool: Optional[str] = None,
        block_layers=Self_Attention_Layer_scale_init_Block,
        block_layers_token=Layer_scale_init_Block_only_token,
        Patch_layer=ConvStem,
        act_layer: nn.Module = nn.GELU,
        Attention_block=Conv_blocks_se,
        dpr_constant: bool = True,
        init_scale: float = 1e-4,
        Attention_block_token_only=Learned_Aggregation_Layer,
        Mlp_block_token_only=Mlp,
        depth_token_only: int = 1,
        mlp_ratio_clstk: float = 3.0,
        multiclass: bool = False,
        mlp_ratio: float = 4.0,
        Self_Attention_num_heads: int = 12,
        Self_Attention_block=Attention,
        Mlp_block=Mlp,
    ):
        super().__init__()

        if flag_fullres:
            self.num_fixations = 1 #5
        else:
            self.num_fixations = 5 #5 #10 #5
        
        if flag_uni and flag_fullres:
            self.pool_layers_K5  = nn.AvgPool2d(5, stride=4)

        #self.pos_embed = nn.Parameter(torch.zeros(1, num_patches, self.embed_dim))
        #self.pos_embed = nn.Parameter(torch.zeros(1, 196, embed_dim))
        self.pos_embeder = PositionalEncodingFourier(dim=embed_dim)

        # Kernel sizes: 3, 5, 7
        self.pool_layers = nn.ModuleList()
        self.pool_layers.append(nn.AvgPool2d(7, stride=7))
        self.pool_layers.append(nn.AvgPool2d(5, stride=4))
        self.pool_layers.append(nn.AvgPool2d(3, stride=3))

        self.multiclass = multiclass
        self.patch_size = patch_size
        self.num_classes = num_classes
        self.num_features = self.embed_dim = embed_dim  # num_features for consistency with other models

        #self.patch_embed = Patch_layer(
        #    img_size=img_size, patch_size=patch_size, in_chans=in_chans, embed_dim=embed_dim
        #)



        if backbone_size=='Medium':
            pref_model = torch_models.resnet18(pretrained=True)
            pref_model.layer2 = nn.Identity()
            pref_model.layer3 = nn.Identity()
            pref_model.layer4 = nn.Identity()
            pref_model.avgpool = nn.Identity()
            pref_model.fc = nn.Identity()
            pref_model.fc = nn.Sequential(nn.Unflatten(1, torch.Size([64,56,56])),
                                          nn.ReLU(),#inplace=True),
                                          nn.Conv2d(64, 384, kernel_size=4, stride=4, padding=0),
                                          #nn.ReLU(),
                                          nn.Flatten(start_dim=2))
            self.patch_embed = pref_model
        elif backbone_size=='Large':
            pref_model = torch_models.resnet18(pretrained=True)
            pref_model.layer3 = nn.Identity()
            pref_model.layer4 = nn.Identity()
            pref_model.avgpool = nn.Identity()
            pref_model.fc = nn.Identity()
            pref_model.fc = nn.Sequential(nn.Unflatten(1, torch.Size([128,28,28])),
                                          nn.ReLU(),#inplace=True),
                                          nn.Conv2d(128, 384, kernel_size=3, stride=2, padding=1),
                                          #nn.ReLU(),
                                          nn.Flatten(start_dim=2))
            self.patch_embed = pref_model
        elif backbone_size=='Small':
            self.patch_embed = Patch_layer(
                img_size=img_size, patch_size=patch_size, in_chans=in_chans, embed_dim=embed_dim
            )

        self.print_debug = False

        if not self.multiclass:
            self.cls_token = nn.Parameter(torch.zeros(1, 1, int(embed_dim)))
        else:
            self.cls_token = nn.Parameter(torch.zeros(1, num_classes, int(embed_dim)))

        if not dpr_constant:
            dpr = [x.item() for x in torch.linspace(0, drop_path_rate, depth)]
        else:
            dpr = [drop_path_rate for i in range(depth)]

        '''
        self.blocks = nn.ModuleList(
            [
                block_layers(
                    dim=embed_dim,
                    drop_path=dpr[i],
                    norm_layer=norm_layer,
                    act_layer=act_layer,
                    Attention_block=Attention_block,
                    init_values=init_scale,
                )
                for i in range(depth)
            ]
        )
        '''

        self.blocks = nn.ModuleList([
            block_layers(
                embed_dim, Self_Attention_num_heads, mlp_ratio=mlp_ratio, qkv_bias=qkv_bias, qk_scale=qk_scale,
                drop=0.0, attn_drop=attn_drop_rate, drop_path=dpr[i], norm_layer=norm_layer,
                act_layer=act_layer,Attention_block=Self_Attention_block,Mlp_block=Mlp_block,init_values=init_scale)
            for i in range(depth)])
        

        
            
        self.blocks_token_only = nn.ModuleList(
            [
                block_layers_token(
                    dim=int(embed_dim),
                    num_heads=num_heads,
                    mlp_ratio=mlp_ratio_clstk,
                    qkv_bias=qkv_bias,
                    qk_scale=qk_scale,
                    drop=drop_rate,
                    attn_drop=attn_drop_rate,
                    drop_path=0.0,
                    norm_layer=norm_layer,
                    act_layer=act_layer,
                    Attention_block=Attention_block_token_only,
                    Mlp_block=Mlp_block_token_only,
                    init_values=init_scale,
                )
                for i in range(depth_token_only)
            ]
        )

        self.norm = norm_layer(int(embed_dim))

        self.total_len = depth_token_only + depth

        self.feature_info = [dict(num_chs=int(embed_dim), reduction=0, module='head')]
        if not self.multiclass:
            self.head = nn.Linear(int(embed_dim), num_classes) if num_classes > 0 else nn.Identity()
        else:
            self.head = nn.ModuleList([nn.Linear(int(embed_dim), 1) for _ in range(num_classes)])

        self.rescale: float = 0.02

        trunc_normal_(self.cls_token, std=self.rescale)
        self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=self.rescale)
            if m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    @torch.jit.ignore
    def no_weight_decay(self):
        return {'cls_token'}

    def get_classifier(self):
        return self.head

    def get_num_layers(self):
        return len(self.blocks)

    def reset_classifier(self, num_classes: int, global_pool: str = ''):
        self.num_classes = num_classes
        self.head = nn.Linear(self.embed_dim, num_classes) if num_classes > 0 else nn.Identity()


    def foveate_feat_V4(self, x, py_idx, px_idx):
        B = x.shape[0]
        if self.print_debug:
            print('foveate_feat_V4: #1: x.shape: {}'.format(x.shape))

        # Reshape into 2D shape
        x = x.reshape(B, 14, 14, -1).permute(0, 3, 1, 2).contiguous()
        if self.print_debug:
            print('foveate_feat_V4: #2: x.shape: {}'.format(x.shape))
        #pool_cnt = 29

        feat_dim = self.embed_dim #192 #256 #1024
        #x = torch.stack([F.pad(img, (13-px_idx[idx], px_idx[idx], 13-py_idx[idx], py_idx[idx])) for idx, img in enumerate(x)])
        x = torch.stack([F.pad(img.unsqueeze(0), (13-px_idx[idx], px_idx[idx], 13-py_idx[idx], py_idx[idx]), "reflect") for idx, img in enumerate(x)]).squeeze()
        if self.print_debug:
            print('foveate_feat_V4: #3: x.shape: {}'.format(x.shape))

        # Pool the input batch
        #x_all = [self.pool_layers[0](x[:, :, 4:23, 4:23]).flatten(2)]
        #x_all += [self.pool_layers[1](x[:, :, 10:17, 10:17]).flatten(2)]
        x_all = [self.pool_layers[0](x[:, :, 3:24, 3:24]).flatten(2)]
        x_all += [self.pool_layers[1](x[:, :, 9:18, 9:18]).flatten(2)]
        #print('x_all[0].shape: {}, x_all[1].shape: {}'.format(x_all[0].shape, x_all[1].shape))

        x_all = torch.cat(x_all, dim=2).permute(0, 2, 1).contiguous()
        #print('x_all.shape: {}'.format(x_all.shape))
        #x_3 = self.pool_layers[2](x[:, :, 12:16, 12:16]).flatten(2).permute(0, 2, 1)
        #print('x_3.shape: {}'.format(x_3.shape))
        '''
        py_idx = torch.tensor(py_idx).cuda().unsqueeze(1)
        px_idx = torch.tensor(px_idx).cuda().unsqueeze(1)
        py_f = py_idx + self.y_diff
        px_f = px_idx + self.x_diff

        valid_idx = torch.logical_and(torch.logical_and(py_f>-1, py_f<14), torch.logical_and(px_f>-1, px_f<14))
        valid_idx = torch.logical_and(valid_idx, self.valid_idx)


        pos_f = py_f*14 + px_f
        val_idx, sorted_idx = torch.sort(valid_idx.cpu().int(), descending=True)

        x_all = x_all[np.arange(B), sorted_idx[:, :pool_cnt].t()].permute(1,0,2).contiguous()
        pos_f = pos_f[np.arange(B), sorted_idx[:, :pool_cnt].t()].permute(1,0).contiguous()
        pos_f = (pos_f.cpu()+10)*val_idx[:,:pool_cnt]-10
        '''
        if self.print_debug:
            print('foveate_feat_V4: #4: x_all.shape: {}'.format(x_all.shape))
        return x_all #, x_3 #, pos_f.numpy().astype(int)


    #def forward_features(self, x: torch.Tensor) -> torch.Tensor:
    def forward(self, images: torch.Tensor) -> torch.Tensor:
        
        B = images.shape[0]
        predictions = torch.zeros(B, self.num_fixations, 1000).cuda()
        B, C, H, W = images.shape
        Hp = H // self.patch_size
        Wp = W // self.patch_size
        if self.print_debug:
            print('forward_features #1: images.shape: {}, Hp: {}, Wp: {}'.format(images.shape, Hp, Wp))
        #x_full = self.patch_embed(x)
        

        x_full = self.patch_embed(images).transpose(1,2) #+ self.pos_embed
        if self.print_debug:
            print('output of patch embedding: x_full.shape: {}'.format(x_full.shape))
        pos_encoding = self.pos_embeder(B, Hp, Wp).reshape(B, -1, x_full.shape[1]).permute(0, 2, 1)
        if self.print_debug:
            print('shape of position encoding: pos_encoding.shape: {}'.format(pos_encoding.shape))
        x_full = x_full + pos_encoding



        xr = x_full.reshape(B,14,14,self.embed_dim)
        #print('forward_features #2: After patch_embed: x.shape: {}'.format(x.shape))
        cls_tokens = self.cls_token.expand(B, -1, -1)
        #print('forward_features #2: After patch_embed: cls_token.shape: {}, cls_tokens.shape: {}'.format(self.cls_token.shape, cls_tokens.shape))
        valid_idx = torch.zeros((B,5,5))
        done_fix = np.zeros((B,196))
        #print('\n\n')
        spat_sz = 14
        alpha_accum = np.zeros([B, spat_sz, spat_sz]) #.cuda()

        plot_xy = []
        x_global = []
        for fix_idx in range(self.num_fixations):
            if fix_idx==0:
                py_idx = np.random.randint(0,spat_sz,[B])#,1])
                px_idx = np.random.randint(0,spat_sz,[B])#,1])
                #py_idx = np.random.randint(6,7,[B])#,1])
                #px_idx = np.random.randint(7,8,[B])#,1])

            plot_xy.append([px_idx[0]*16+8, py_idx[0]*16+8])
            
            if flag_FR==1:
                IOR_h, IOR_w = [0], [0]
            else:
                IOR_h, IOR_w = [-1, 0, 1], [-1, 0, 1]
            
            for h_ in IOR_h:
                for w_ in IOR_w:
                    for df_idx in range(B):
                        cy_ = py_idx[df_idx]+h_
                        cx_ = px_idx[df_idx]+w_
                        if cy_>-1 and cx_>-1 and cy_<spat_sz and cx_<spat_sz:  #try:
                            df = cy_*spat_sz+cx_
                            done_fix[df_idx, int(df)] = 16
            if flag_FR>1:
                py_idx = np.clip(py_idx, 1, spat_sz-2)
                px_idx = np.clip(px_idx, 1, spat_sz-2)
                #for df_idx in range(B):

            if not flag_fullres:
                #x_p, x_3 = self.foveate_feat_V4(x_full[:,1:,:], py_idx, px_idx)
                #x_p = self.foveate_feat_V4(x_full[:,1:,:], py_idx, px_idx)
                x_p = self.foveate_feat_V4(x_full, py_idx, px_idx)
                #print('done with pooling!!!!!!, x_p.shape: {}'.format(x_p.shape))

                'Fovea 1x1, Retain 1x1'
                if flag_FR==1: #'''
                    x_fov_ = torch.stack([xr[im, py_idx[im], px_idx[im], :].reshape(1,self.embed_dim)    for im in range(B)], dim=0)
                    #x_global.append( torch.stack((x_fov_,x_3), dim=1).squeeze(2) )
                    #x_fov = torch.cat( (x_p, x_fov_), dim=1)
                    x_global.append(x_fov_)
                    x_fov = torch.cat( (x_p,torch.cat(x_global, dim=1)), dim=1)
                'Fovea 3x3, Retain 3x3'
                if flag_FR==3: #'''
                    x_fov_ = torch.stack([xr[im, (py_idx[im]-1):(py_idx[im]+2), (px_idx[im]-1):(px_idx[im]+2), :].reshape(9,self.embed_dim)    for im in range(B)], dim=0)
                    #x_fov = torch.cat( (x_p, x_fov_), dim=1)
                    x_global.append(x_fov_)
                    x_fov = torch.cat( (x_p,torch.cat(x_global, dim=1)), dim=1)

            else:
                if flag_uni:
                    x_fov = x_full_5
                else:
                    x_fov = x_full #[:,1:]

            #if fix_idx==0:
            #    c_vec = x_full[:,0,:].unsqueeze(1)

            #x = torch.cat((c_vec, x_fov), dim=1)
            #x = torch.cat((x_full[:,0,:].unsqueeze(1), x_fov), dim=1)
            #print('x.shape: {}'.format(x.shape))
            #print('fix_idx: {}, x.shape: {}'.format(fix_idx, x.shape))          
            x = x_fov
            if self.print_debug:
                print('forward: before blocks: x.shape: {}'.format(x.shape))
            for i, blk in enumerate(self.blocks):
                x = blk(x)
            #print('forward_features #3: After blocks: x.shape: {}'.format(x.shape))
            if self.print_debug:
                print('forward: after blocks: x.shape: {}'.format(x.shape))

            for i, blk in enumerate(self.blocks_token_only):
                cls_tokens, attn = blk(x, cls_tokens)
            if self.print_debug:
                print('forward: after blocks_token_only: x.shape: {}, cks_tokens.shape: {}, attn.shape: {}'.format(x.shape, cls_tokens.shape, attn.shape))
            x = torch.cat((cls_tokens, x), dim=1)
            if self.print_debug:
                print('forward: after concat: x.shape: {}, cks_tokens.shape: {}, attn.shape: {}'.format(x.shape, cls_tokens.shape, attn.shape))
            #print('forward_features #4: After blocks_token_only: cls_tokens.shape: {}, x.shape: {}'.format(cls_tokens.shape, x.shape))

            #c_vec = x[:,0,:].unsqueeze(1)
            x = self.norm(x)
            if self.print_debug:
                print('forward: after normalization: x.shape: {}'.format(x.shape))


            x = x[:, 0]
            if self.print_debug:
                print('forward: before head: x.shape: {}'.format(x.shape))
            #x = self.head(x)


            predictions[:, fix_idx] = self.head(x)


            if self.save_images:
                preds = predictions[:, fix_idx]
                class_idx = torch.max(preds[0], 0)[1]
                predicted_label = str(class_idx.cpu().item()) #self.class_names[class_idx]
                predicted_prob = F.softmax(preds[0], dim=0)[class_idx]
                if fix_idx>0:
                    fix_list = fix_list + '\n -->'
                else:
                    fix_list = ''
                fix_list = fix_list + '(' + class_names[str(predicted_label)][1] + ', ' + str(np.floor(predicted_prob.item()*100)/100) + ')'




            if not flag_fullres and fix_idx<(self.num_fixations-1):
                attn = attn[:, 0, 0, 1:].detach().cpu().numpy() #torch.mean(attn_weights[fix_idx], 1)[:,0,1:].detach().cpu().numpy()
                if self.print_debug:
                    print('#1: attn.shape: {}'.format(attn.shape))
                attn /= np.sum(attn, axis=1, keepdims=True)
                #print('#2: attn: {}'.format(np.sum(attn, axis=1)))
                attn_7 = attn[:,0:9].reshape(-1,3,3)
                attn_5 = attn[:,9:13].reshape(-1,2,2)#(9+max_cnt)] #.reshape(-1,3,3)
                #alpha_accum = block_reduce(np.repeat(np.repeat(attn_7,6,axis=1),6,axis=2), block_size=(1,3,3), func=np.mean)
                alpha_accum_7 = np.repeat(np.repeat(attn_7,7,axis=1),7,axis=2)#[:, 1:-1, 1:-1]
                alpha_accum_5 = np.repeat(np.repeat(attn_5,5,axis=1),5,axis=2)[:, :-1, :-1]
                #print('Step 1: alpha_accum_7.shape: {}, alpha_accum_5.shape: {}, alpha_accum.shape: {}'.format(alpha_accum_7.shape, alpha_accum_5.shape, alpha_accum.shape))
                pad_7  = [0,0,0,1,1,1, 1,1,1,2,2,2, 2,2,2,3,3,3]
                pad_7_ = [0,0,0,1,1,1, 1,1,1,2,2,2, 2,2,2,3,3,3]
                pad_5  = [0,0,1,1, 1,2,2]
                pad_5_ = [0,0,1,1, 1,2,2]
                'unequal inserts in x,y to make it assymmetric'
                alpha_accum_7 += 0.05*np.insert(np.insert(attn_7, pad_7, 0, axis=1), pad_7_, 0, axis=2)
                alpha_accum_5 += 0.05*np.insert(np.insert(attn_5, pad_5, 0, axis=1), pad_5_, 0, axis=2)
                ''' NEED TO ADD AVERAGING BETWEEN POOLING REGIONS'''
                alpha_accum_7[:, 6:15, 6:15] = alpha_accum_5
                cz = 10 
                ' Decrease the attention values '
                alpha_accum *= 0.5
                #print('Step 2: alpha_accum_7.shape: {}, alpha_accum_5.shape: {}, alpha_accum.shape: {}'.format(alpha_accum_7.shape, alpha_accum_5.shape, alpha_accum.shape))
                for img_idx in range(B):
                    ''' Center of the attention map is at 10'''
                    fy, fx = py_idx[img_idx], px_idx[img_idx]
                    #print('idx: {}/{}, fixation: ({}, {})'.format(img_idx, B, fy, fx))
                    L, R = np.minimum(fx, cz), np.minimum(13-fx, cz)+1
                    T, B_ = np.minimum(fy, cz), np.minimum(13-fy, cz)+1
                    #print('Limits: T:{}, B_:{}, L:{}, R:{}'.format(T, B_, L, R))
                    alpha_accum[img_idx, (fy-T):(fy+B_), (fx-L):(fx+R)] += alpha_accum_7[img_idx, (cz-T):(cz+B_), (cz-L):(cz+R)]
                pos_idx = np.argmax(alpha_accum.reshape(-1,spat_sz*spat_sz)-done_fix, 1)
                #print('alpha_accum_7[0]: {}'.format((alpha_accum_7[0]*1000).astype(int)))
                #print('alpha_accum[0]: {}'.format((alpha_accum[0]*1000).astype(int)))
                py_idx = pos_idx//spat_sz
                px_idx = pos_idx%spat_sz

                    

                    

            if self.save_images and fix_idx==(self.num_fixations-1): # a00==0:
                fig, ax = plt.subplots() 
                img = images[0].cpu() #transpose(0,1).transpose(1,2).cpu() #img.permute(1,2,0).cpu()
                img = inv_transform(img)
                ax.imshow(img)
                # Plot the fixations now
                for t in range(fix_idx+1): #len(plot_xy)):
                    ax.scatter(plot_xy[t][0], plot_xy[t][1], c='tab:blue', s=40*t+20, label=str(t), alpha=0.5)
                    if t>0:
                        #time.sleep(1)
                        ax.arrow(plot_xy[t-1][0], plot_xy[t-1][1], plot_xy[t][0]-plot_xy[t-1][0], plot_xy[t][1]-plot_xy[t-1][1], head_width=5, head_length=10, fc='r', ec='r', length_includes_head=True)
                #ax.title.set_text('{} ({}): {}'.format(str(self.target[0].item()), class_th, fix_list)) #, wrap=True, fontsize=10)
                #ax.title.set_text('{}: {}'.format(class_names[str(self.target[0].item())][1], fix_list)) #, wrap=True, fontsize=10)
                ax.title.set_text('{} : {}'.format(class_names[str(self.target[0].item())][1], fix_list)) #, wrap=True, fontsize=10)
                ax.axis('off')
                plt.axis('off')
                plt.show()
                #plt.savefig('saved_images/' + str(self.batch_idx) + '_' + str(fix_idx) +'.png', bbox_inches='tight',pad_inches = 0, dpi = 200)
                save_folder = 'saved_images/' + str(self.epoch) + '/' #+str(fix_idx+1)+'/'
                if False: #class_idx==target_idx:
                    plt.savefig(save_folder + 'TP/' + str(self.batch_idx) + '.png', bbox_inches='tight',pad_inches = 0, dpi = 200)
                #else:
                if True: #class_idx!=target_idx:
                    plt.savefig(save_folder + str(self.batch_idx) + '.png', bbox_inches='tight',pad_inches = 0, dpi = 200)
                #plt.savefig('saved_images/' + str(self.batch_idx) + '.png', bbox_inches='tight',pad_inches = 0, dpi = 200)
                plt.cla()
                plt.close()

        predictions = predictions.transpose(1,2)

        return predictions #[:, -1]





    '''
        return x

        if not self.multiclass:
            return x[:, 0]
        else:
            return x[:, : self.num_classes].reshape(B, self.num_classes, -1)

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        B = x.shape[0]
        predictions = torch.zeros(B, self.num_fixations, 1000).cuda()
        #print('\nforward #1: x.shape: {}'.format(x.shape))
        x = self.forward_features(x)
        #print('forward #2: x.shape: {}'.format(x.shape))
        if not self.multiclass:
            x = self.head(x)
            return x
        else:
            all_results = []
            for i in range(self.num_classes):
                all_results.append(self.head[i](x[:, i]))
            return torch.cat(all_results, dim=1).reshape(B, self.num_classes)
    '''

@register_model
def S60(pretrained: bool = False, **kwargs):
    model = PatchConvnet(
        patch_size=16,
        embed_dim=384,
        depth=12,#60,
        num_heads=1,
        qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6),
        Patch_layer=ConvStem,
        Attention_block=Conv_blocks_se,
        depth_token_only=1,
        mlp_ratio_clstk=3.0,
        **kwargs
    )

    return model


@register_model
def S120(pretrained: bool = False, **kwargs):
    model = PatchConvnet(
        patch_size=16,
        embed_dim=384,
        depth=120,
        num_heads=1,
        qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6),
        Patch_layer=ConvStem,
        Attention_block=Conv_blocks_se,
        init_scale=1e-6,
        mlp_ratio_clstk=3.0,
        **kwargs
    )

    return model


@register_model
def B60(pretrained: bool = False, **kwargs):
    model = PatchConvnet(
        patch_size=16,
        embed_dim=768,
        depth=60,
        num_heads=1,
        qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6),
        Attention_block=Conv_blocks_se,
        init_scale=1e-6,
        **kwargs
    )

    return model


@register_model
def B120(pretrained: bool = False, **kwargs):
    model = PatchConvnet(
        patch_size=16,
        embed_dim=768,
        depth=120,
        num_heads=1,
        qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6),
        Patch_layer=ConvStem,
        Attention_block=Conv_blocks_se,
        init_scale=1e-6,
        **kwargs
    )

    return model


@register_model
def L60(pretrained: bool = False, **kwargs):
    model = PatchConvnet(
        patch_size=16,
        embed_dim=1024,
        depth=60,
        num_heads=1,
        qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6),
        Patch_layer=ConvStem,
        Attention_block=Conv_blocks_se,
        init_scale=1e-6,
        mlp_ratio_clstk=3.0,
        **kwargs
    )

    return model


@register_model
def L120(pretrained: bool = False, **kwargs):
    model = PatchConvnet(
        patch_size=16,
        embed_dim=1024,
        depth=120,
        num_heads=1,
        qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6),
        Patch_layer=ConvStem,
        Attention_block=Conv_blocks_se,
        init_scale=1e-6,
        mlp_ratio_clstk=3.0,
        **kwargs
    )

    return model


@register_model
def S60_multi(pretrained: bool = False, **kwargs):
    model = PatchConvnet(
        patch_size=16,
        embed_dim=384,
        depth=60,
        num_heads=1,
        qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6),
        Patch_layer=ConvStem,
        Attention_block=Conv_blocks_se,
        Attention_block_token_only=Learned_Aggregation_Layer_multi,
        depth_token_only=1,
        mlp_ratio_clstk=3.0,
        multiclass=True,
        **kwargs
    )

    return model
