from datetime import datetime
import json, re, os, sys, time
import boto3
from collections import defaultdict
from LC_Agent.utils import read_json, split_text_into_keywords, convert_to_tokenizer_format
from rank_bm25 import BM25Okapi
from copy import deepcopy


class StateManager:
    def __init__(self): 
        self.notes = {}
        print("[INFO] StateManager Initialized")
    
    def add_note(self, key, content, summary): 
        self.notes[key] = {"summary": summary, "full_content": content}
        # print(f"[ACTION]: Note added: Key='{key}', Summary='{summary}'")
        return {"status": "success", "key": key}
    
    def read_note(self, key): 
        # print(f"[ACTION] Reading note: Key='{key}'")
        return self.notes.get(key, {"error": "Note not found"})
    
    def update_note(self, key, new_content=None, new_summary=None):
        """Update or delete a note by key."""
        if key not in self.notes:
            return {"error": "Note not found"}
        
        if new_content == "" and new_summary == "":
            del self.notes[key]
            # print(f"[ACTION]: Note deleted: Key='{key}'")
            return {"status": "success", "key": key, "message": "Note deleted."}
        
        if new_content is not None:
            self.notes[key]["full_content"] = new_content
        if new_summary is not None:
            self.notes[key]["summary"] = new_summary
        # print(f"[ACTION]: Note updated: Key='{key}'")
        return {"status": "success", "key": key, "message": "Note updated."}
    
    def merge_notes(self, keys, new_key, new_summary):
        """Merge Multiple Notes into One."""
        contents = {}
        merged_keys = []
        for key in keys:
            if key in self.notes:
                contents[key] = self.notes[key]["full_content"]
                merged_keys.append(key)
                del self.notes[key]
        
        if merged_keys:
            merged_content = {"merged_from": merged_keys, "content": contents}
            self.notes[new_key] = {"summary": new_summary, "full_content": merged_content}
            # print(f"[ACTION]: Notes merged: {merged_keys} -> {new_key}")
            return {"status": "success", "new_key": new_key, "merged_keys": merged_keys}
        return {"error": "No notes found to merge."}
    
    def get_notes_summary(self):
        # print("[ACTION]: Getting notes summary")
        if not self.notes: return "No notes recorded."
        return "\n".join([f"- **{key}**: {data['summary']}" for key, data in self.notes.items()])

class ToolLibrary:
    def __init__(self, state_manager, tokenizer, document_content):
        self.state_manager = state_manager
        self.document = document_content
        self.tokenizer = tokenizer
        self.chunk_pointer = [-1, 0]  # (next_forward_chunk_id, next_backward_chunk_id)
        self.index = []
        self._bm25_corpus = None
        self.keyword_index = {}
        self.encoded_doc = self.tokenizer(self.document, return_offsets_mapping=True, add_special_tokens=False)
        print("[INFO] ToolLibrary Initialized")

    def analyzeText(self, params):
        return {
            "file_name": "merged_document.txt", 
            "total_tokens": len(self.encoded_doc["input_ids"])
        }

    def buildIndex(self, params):
        chunk_size = params.get("chunk_size", 4000)
        overlap = params.get("overlap", 200)
        
        if chunk_size > 8000:
            return {"error": "chunk_size exceeds 8000 tokens limit."}

        input_ids = self.encoded_doc["input_ids"]
        offsets = self.encoded_doc["offset_mapping"]

        self.index = []
        self._bm25_corpus = []
        start_token = 0
        chunk_id = 0
        while start_token < len(input_ids):
            end_token = min(start_token + chunk_size, len(input_ids))

            # Get corresponding character positions
            chunk_offsets = offsets[start_token:end_token]
            char_start = chunk_offsets[0][0]
            char_end = chunk_offsets[-1][1]

            # Extract raw text based on char span
            chunk_content = self.document[char_start:char_end]
            chunk_tokens = split_text_into_keywords(chunk_content)
            self._bm25_corpus.append(chunk_tokens)
            chunk_data = {
                "chunk_id": chunk_id,
                "content": chunk_content,
                "start_pos": start_token,
                "end_pos": end_token
            }
            self.index.append(chunk_data)

            chunk_id += 1
            start_token += chunk_size - overlap
        
        self.keyword_index = defaultdict(list)
        self._bm25_corpus = BM25Okapi(self._bm25_corpus)
        self.chunk_pointer = [-1, 0]  # reset after (re)build

        return {
            "index_id": "document_index",
            "total_chunks": len(self.index),
            "first_chunk_id": 0,
            "last_chunk_id": len(self.index) - 1,
        }

    def loadDocument(self, params):
        """Load the full document content."""
        if not self.document:
            return {"error": "Document content is empty."}
        return {
            "document_content": self.document
        }
    
    def readChunk(self, params):
        chunk_id = params.get("chunk_id", 0)
        try:
            chunk_id = int(chunk_id)
        except (ValueError, TypeError):
            return {"error": "chunk_id must be an integer."}
        if chunk_id < 0 or chunk_id > (len(self.index)-1):
            return {"error": f"Chunk_id: {chunk_id} is out of range. It must be between 0 and {len(self.index)-1}."}
        return {"retrieved_chunk": [self.index[chunk_id]], "chunk_id": chunk_id}

    def nextChunk(self, params):
        order = params.get("order", 'forward')
        if order not in ['forward', 'backward']:
            return {"error": "Order must be either 'forward' or 'backward'."}
        total_chunks = len(self.index)
        if total_chunks == 0:
            return {"error": "Index not built. Please call 'buildIndex' first."}

        if order == 'forward':
            next_id = self.chunk_pointer[0] + 1
            if next_id >= len(self.index):
                return {"error": "No more chunks available in forward direction."}
            self.chunk_pointer[0] = next_id
            forward_progress = f"{next_id+1}/{total_chunks}" # next_id=0 means read one from forward
            backward_progress = f"{abs(self.chunk_pointer[1])}/{total_chunks}" # next_id=-1 means read one from backward
            return {"retrieved_chunk": [self.index[self.chunk_pointer[0]]], "chunk_id": next_id, "reading_progress": {"forward": forward_progress, "backward": backward_progress}}
        elif order == 'backward':
            next_id = self.chunk_pointer[1] - 1
            if next_id < -len(self.index):
                return {"error": "No more chunks available in backward direction."}
            self.chunk_pointer[1] = next_id
            forward_progress = f"{self.chunk_pointer[0]+1}/{total_chunks}" # next_id=0 means read one from forward
            backward_progress = f"{abs(next_id)}/{total_chunks}" # next_id=-1 means read one from backward
            return {"retrieved_chunk": [self.index[self.chunk_pointer[1]]], "chunk_id": len(self.index) + next_id, "reading_progress": {"forward": forward_progress, "backward": backward_progress}}

    def searchEngine(self, params):
        raw_kw      = params.get("keyword", "")
        
        # Check if index is built
        if not self.index or self._bm25_corpus is None:
            return {"error": "Index not built. Please call buildIndex first."}
        
        keywords = [k.strip().lower() for k in raw_kw.split(",") if k.strip()]
        if not keywords:
            return {"error": "keyword cannot be empty."}

        keywords_as_key = "__".join(keywords)
        if keywords_as_key in self.keyword_index:
            chunk_scores = self.keyword_index[keywords_as_key]
        else:
            keyword_tokens = []
            for kw in keywords:
                keyword_tokens.extend(split_text_into_keywords(kw))
            print(f"[INFO] Calculating BM25 scores for keywords '{keyword_tokens}'...")
            chunk_scores = self._bm25_corpus.get_scores(keyword_tokens)
            chunk_scores = [round(score, 3) for score in chunk_scores]
            self.keyword_index[keywords_as_key] = chunk_scores

        THRESHOLD = 0.0 # maybe we can ask the model to choose the threshold?
        valid_matches = [i for i in range(len(chunk_scores)) if chunk_scores[i] > THRESHOLD]
        total_matches = len(valid_matches)
        # total_pages   = (total_matches + per_page - 1) // per_page
        if total_matches == 0:
            return {
                "retrieved_chunks": [],
                "message": "No matching content found.",
                "keywords": keywords,
            }

        chunks = []
        for idx in valid_matches[:]:
            chunk = deepcopy(self.index[idx])
            del chunk["content"]
            del chunk["start_pos"]
            del chunk["end_pos"]
            chunk["relevance_score"] = chunk_scores[idx]
            chunks.append(chunk)
        
        # Limit to top 20 chunks if there are more than 20 matches
        if len(chunks) > 20:
            chunks.sort(key=lambda x: x["relevance_score"], reverse=True)
            chunks = chunks[:20]
            message = f"Showing the most relevant 20/{total_matches} chunks."
            return {
                "retrieved_chunks": chunks,
                "message": message,
                "keywords": keywords,
            }
        else:
            return {
                "retrieved_chunks": chunks,
                "keywords": keywords
            }

    def checkBudget(self, params):
        raise NotImplementedError

    def note(self, params):
        return self.state_manager.add_note(params['key'], params['content'], params.get('summary', 'No summary provided.'))

    def readNote(self, params):
        return self.state_manager.read_note(params['key'])

    def updateNote(self, params):
        return self.state_manager.update_note(
            params['key'],
            params.get('new_content'),
            params.get('new_summary')
        )

    def mergeNotes(self, params):
        return self.state_manager.merge_notes(
            params['keys'],
            params['new_key'], 
            params['new_summary']
        )

    def deleteContext(self, params):
        """Delete the content of a specific tool call."""
        tool_id = params.get("tool_id")
        if tool_id:
            return {"status": "success", "deleted_tool_id": tool_id, "message": f"Tool {tool_id} content has been deleted."}
        return {"error": "tool_id is required."}

    def getContextStats(self, params):
        """Get context statistics."""
        return {
            "total_notes": len(self.state_manager.notes),
            "notes_keys": list(self.state_manager.notes.keys()),
            "index_chunks": len(self.index),
            "document_size": len(self.encoded_doc["input_ids"]),
            "searched_keywords": list(self.keyword_index.keys()),
        }

    def finish(self, params):
        return {"final_answer": params.get("answer", "No final answer provided.")}


class ExecLogger:
    """Execution Logger for saving query logs and inference results"""
    def __init__(self, log_dir="logs", results_dir="results"):
        self.log_dir = log_dir
        self.results_dir = results_dir
        self.ensure_output_dir()
        self.session_id = datetime.now().strftime("%Y%m%d_%H%M%S")
        self.api_calls = []
        
    def ensure_output_dir(self):
        if not os.path.exists(self.log_dir):
            os.makedirs(self.log_dir)
        if not os.path.exists(self.results_dir):
            os.makedirs(self.results_dir)

    def log_api_call(self, api_input, api_output, call_index):
        api_call = {
            "timestamp": datetime.now().isoformat(),
            "call_index": call_index,
            "session_id": self.session_id,
            "api_input": api_input,
            "api_output": api_output
        }
        self.api_calls.append(api_call)
        print(f"[INFO] API call {call_index} has been recorded")
    
    def save_query_log(self, query, document_info, timestamp=None):
        if timestamp is None:
            timestamp = datetime.now().isoformat()
        
        log_entry = {
            "timestamp": timestamp,
            "session_id": self.session_id,
            "query": query,
            "document_info": document_info,
            "api_calls_count": len(self.api_calls)
        }
        
        log_file = os.path.join(self.log_dir, f"query_log_{self.session_id}.json")
        
        if os.path.exists(log_file):
            with open(log_file, 'r', encoding='utf-8') as f:
                logs = json.load(f)
        else:
            logs = []
        
        logs.append(log_entry)
        
        with open(log_file, 'w', encoding='utf-8') as f:
            json.dump(logs, f, ensure_ascii=False, indent=2)

        print(f"[INFO] Query log saved to: {log_file}")

    def save_inference_result(self, query, orchestrator, result_info=None, prefix_tag=None):
        timestamp = datetime.now().isoformat()
        
        inference_trace = {
            "timestamp": timestamp,
            "session_id": self.session_id,
            "system_prompt": orchestrator.system_prompt,
            "query": query,
            "result_info": result_info or {},
            "full_history": orchestrator.full_history
        }
        
        for i, msg in enumerate(orchestrator.full_history):
            if msg["role"] == "assistant":
                for block in msg.get("content", []):
                    if block.get("type") == "tool_use":
                        tool_call = {
                            "tool_id": block.get("id"),
                            "tool_name": block.get("name"),
                            "tool_params": block.get("input"),
                            "position": i
                        }
            
            elif msg["role"] == "tool":
                tool_result = {
                    "tool_id": msg.get("tool_id"),
                    "tool_name": msg.get("tool_name"),
                    "tool_use_id": msg.get("tool_use_id"),
                    "result": msg.get("content"),
                    "position": i
                }
        
        result_file = os.path.join(self.log_dir, f"inference_result_{self.session_id}.json")
        if prefix_tag:
            result_file = os.path.join(self.log_dir, f"{prefix_tag}_inference_result_{self.session_id}.json")
        with open(result_file, 'w', encoding='utf-8') as f:
            json.dump(inference_trace, f, ensure_ascii=False, indent=2)

        print(f"[INFO] Inference results saved to: {result_file}")
        return result_file
    
    def save_api_calls_log(self):
        """Save API calls log separately"""
        if not self.api_calls:
            return
        
        api_log_file = os.path.join(self.log_dir, f"api_calls_{self.session_id}.json")
        with open(api_log_file, 'w', encoding='utf-8') as f:
            json.dump(self.api_calls, f, ensure_ascii=False, indent=2)

        print(f"[INFO] API calls log saved to: {api_log_file}")

    def save_final_result(self, orchestrator, question, expected_answer, meta_info=None, filename=None):
        """Save the final results and metadata to the results directory"""
        full_history = orchestrator.full_history
        final_answer = None
        for msg in reversed(full_history):
            if msg.get("role") == "tool":
                final_answer = msg.get("content", {}).get("final_answer")
                if final_answer:
                    break
        if not final_answer:
            final_answer = "No final answer found."
        result_info = {
            "session_id": self.session_id,
            "question": question,
            "final_answer": final_answer,
            "expected_answer": expected_answer,
            "meta_info": meta_info or {}
        }
        if filename:
            result_file = os.path.join(self.results_dir, filename)
        else:
            if meta_info:
                sample_id = meta_info.get("sample_id", "id_unknown")
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            result_file = os.path.join(self.results_dir, f"{sample_id}_final_result_{timestamp}.json")

        with open(result_file, 'w', encoding='utf-8') as f:
            json.dump(result_info, f, ensure_ascii=False, indent=4)
        print(f"[INFO] Final results saved to: {result_file}")
        return result_file, final_answer

class Orchestrator:
    def __init__(self, 
                 claude_config, 
                 document_content, 
                 temperature, 
                 tokenizer, 
                 logger=None, 
                 max_context_exp=30720, 
                 max_turns_exp=50, 
                 max_output_tokens=4096, 
                 system_prompt_name=None, 
                 tool_config_path=None
                ):
        print("[INFO] Setting up Claude (Bedrock) Client...")
        self.claude_config = claude_config
        claude_access_key = claude_config.get("CLAUDE_ACCESS_KEY")
        claude_secret_key = claude_config.get("CLAUDE_SECRET_KEY")
        claude_region = claude_config.get("CLAUDE_REGION")
        claude_model_id = claude_config.get("CLAUDE_MODEL_ID")

        self.claude_client = boto3.client(
            'bedrock-runtime',
            aws_access_key_id=claude_access_key,
            aws_secret_access_key=claude_secret_key,
            region_name=claude_region
        )
        self.claude_model_id = claude_model_id
        self.system_prompt = self._get_system_prompt_text(system_prompt_name)
        print(f"[INFO] System prompt set as '{self.system_prompt}'")
        self.tools = self._get_tool_config(tool_config_path)
        print(f"[INFO] Claude Client has been set up for model '{claude_model_id}' with {len(self.tools)} tools configured.")

        self.state_manager = StateManager()
        self.tool_library = ToolLibrary(self.state_manager, tokenizer, document_content)
        self.tokenizer = tokenizer
        self.full_history = []
        # self.deleted_tool_ids = set()
        # self.tool_call_counter = 0
        self.ctx_counter = 0  # unified counter for assistant/tool messages
        self.deleted_msg_ids = set()   # contains msg_id of both assistant & tool messages

        self.logger = logger
        self.api_call_counter = 0
        self.temperature = temperature
        self.max_context_exp = max_context_exp
        self.max_turns = max_turns_exp
        self.max_output_tokens = max_output_tokens

    def _get_system_prompt_text(self, system_prompt_name=None):
        if system_prompt_name is None:
            system_prompt_name = "CLAUDE_SYSTEM_PROMPT_7_OP"
        from LC_Agent import prompts
        system_prompt = getattr(prompts, system_prompt_name)
        print(f"[INFO] Using system prompt: {system_prompt_name}")
        return system_prompt

    def _get_tool_config(self, tool_config_path=None):
        if tool_config_path:
            print(f"[INFO] Using custom tool config: {tool_config_path}")
            return read_json(tool_config_path)
        print(f"[INFO] Using default tool config: tools_qwen_full.json")
        return read_json("./LC_Agent/tools_qwen_full.json")

    def _call_claude_api(self, api_payload):
        print("\n" + "="*50 + "\n[API] Calling CLAUDE API...")
        print(f"[API] Number of messages: {len(api_payload)}")
        body = {
            "anthropic_version": "bedrock-2023-05-31",
            "max_tokens": self.max_output_tokens,
            "system": self.system_prompt,
            "messages": api_payload,
            "tools": self.tools,
            "temperature": self.temperature
        }

        max_retries = 3
        call_times = 0
        while True:
            try:
                response = self.claude_client.invoke_model(
                    body=json.dumps(body), modelId=self.claude_model_id
                )
                time.sleep(1)
                response_body = json.loads(response.get('body').read())

                self.api_call_counter += 1
                if self.logger:
                    self.logger.log_api_call(body, response_body, self.api_call_counter)
                
                return response_body
            except Exception as e:
                print(f"[API] Error: Calling Claude API failed: {e}")
                print("--- API Payload DUMP ---")
                print(json.dumps(body, indent=2, ensure_ascii=False))
                print("--- END DUMP ---")
                if call_times < max_retries:
                    sleep_time = 10 * (2 ** call_times)
                    print(f"[INFO] Waiting {sleep_time} seconds before retrying...")
                    time.sleep(sleep_time)
                    call_times += 1
                    continue
                print(f"[INFO] Reached maximum retry limit {max_retries}, stopping retries.")
                error_response = {"stop_reason": "error", "content": [{"type": "text", "text": f"Error occurred while calling API: {e}.", "latest_payload": body["messages"]}]}
                self.api_call_counter += 1
                if self.logger:
                    self.logger.log_api_call(body, error_response, self.api_call_counter)
                return error_response

    def _resolve_msg_entry(self, msg_id: int):
        for i, m in enumerate(self.full_history):
            if m.get("msg_id") == msg_id:
                return i, m
        return None, None

    def _parse_llm_output(self, response_body):
        thought = ""
        tool_act = None
        tool_params = None
        tool_use_id = None
        stop_reason = response_body.get("stop_reason")

        if response_body.get("content"):
            for block in response_body["content"]:
                if block['type'] == 'text':
                    thought += block['text']
                elif block['type'] == 'tool_use':
                    tool_use_id = block['id']
                    tool_act = block['name']
                    tool_params = block['input']
        
        return thought, tool_act, tool_params, tool_use_id, stop_reason

    def _build_api_payload(self):
        api_history = []
        notes_summary_section = f"\n\n### Notes Summary\n<notes_summary>\n{self.state_manager.get_notes_summary()}\n</notes_summary>"

        for i, msg in enumerate(self.full_history):
            if msg["role"] == "user":
                content = msg["content"]
                if i == 0: content += notes_summary_section
                api_history.append({"role": "user", "content": [{"type": "text", "text": content}]})
            elif msg["role"] == "assistant":
                msg_id = msg["msg_id"]
                if msg_id in self.deleted_msg_ids:
                    stub = {}
                    for block in msg["content"]:
                        if block['type'] == 'tool_use':
                            stub = {
                                "type": "tool_use",
                                "id": block['id'],
                                "name": block['name'],
                                "input": {"message": "Content has been deleted to save space."}
                            }
                            break
                    if stub:
                        api_history.append({"role": "assistant", "content": [{"type": "text", "text": "Content has been deleted to save space."}, stub]})
                    else:
                        api_history.append({"role": "assistant", "content": [{"type": "text", "text": "Content has been deleted to save space."}]})
                else:
                    api_history.append({"role": "assistant", "content": msg["content"]})
            elif msg["role"] == "tool":
                msg_id = msg["msg_id"]
                msg_id_ia = msg["msg_id(invoking_assistant)"]
                tool_use_id = msg["tool_use_id"]
                tool_result_content = msg["content"]
                tool_result_content["msg_id"] = msg_id    # we need to ensure msg_id is included in the result
                tool_result_content["msg_id(invoking_assistant)"] = msg_id_ia
                tool_result_content_cp = deepcopy(tool_result_content)  # 深拷贝以避免修改原始内容
                if msg_id in self.deleted_msg_ids:
                    print(f"[INFO] Attempting to delete {msg.get('tool_name', 'unknown')}")
                    tool_result_content_cp = {
                        "msg_id": msg_id,
                        "msg_id(invoking_assistant)": msg_id_ia,
                        "status": "success",
                        "message": "Content has been deleted to save space.",
                        "original_tool": msg.get("tool_name", "unknown")
                    }
                    if msg.get("tool_name") == "nextChunk":
                        tool_result_content_cp["reading_progress"] = msg["content"]["reading_progress"]
                api_history.append({
                    "role": "user",
                    "content": [{"type": "tool_result", "tool_use_id": tool_use_id, "content": json.dumps(tool_result_content_cp, ensure_ascii=False)}]
                })
        return api_history

    def _execute_tool(self, action, params):
        if action == "checkBudget":
            api_payload = self._build_api_payload()
            messages = convert_to_tokenizer_format(api_payload)
            tokenized_messages = self.tokenizer.apply_chat_template(messages, add_generation_prompt=False, tokenize=True)
            # formatted_messages = self.tokenizer.apply_chat_template(messages, add_generation_prompt=False, tokenize=False)
            # with open("./message_debug.txt", "a") as f:
            #     f.write(f"Tokenized messages: {formatted_messages}\n")
            conv_rounds = len(self.full_history) // 2
            message_len = len(tokenized_messages) + 1500 # tool specs + system prompt buffer
            return {
                "conv_rounds": conv_rounds,
                "available_tokens": self.max_context_exp - message_len - self.max_output_tokens,
                "available_rounds": self.max_turns - conv_rounds, # expected rounds to finish
            }
        if action == "deleteContext":
            msg_id = params.get("msg_id")
            if msg_id is None:
                return {"error": "msg_id is required"}
            idx, entry = self._resolve_msg_entry(int(msg_id))
            if entry is None:
                return {"error": f"msg_id {msg_id} not found"}
            role = entry.get("role")
            if role == "user":
                return {"error": "Deleting user messages is not supported"}
            elif role == "assistant" or role == "tool":
                self.deleted_msg_ids.add(int(msg_id))
                return {"status": "success", "deleted_msg_id": int(msg_id), "deleted_role": role}
            return {"error": f"Unsupported role '{role}' for deletion"}

        if hasattr(self.tool_library, action):
            return getattr(self.tool_library, action)(params or {})
        return {"error": f"Tool '{action}' not found."}

    def run(self, user_query, max_turns_to_fail=80):
        self.full_history.append({"role": "user", "content": user_query})
        
        turn = 1
        self.ctx_counter = 0
        while turn <= max_turns_to_fail: 
            print(f"\n--- Round {turn} (Max {max_turns_to_fail} rounds, expected within {self.max_turns} rounds) ---")
            api_payload = self._build_api_payload()
            response_body = self._call_claude_api(api_payload)
            self.ctx_counter += 1
            thought, action, params, tool_use_id, stop_reason = self._parse_llm_output(response_body)
            msg_id = self.ctx_counter

            self.full_history.append({"role": "assistant", "content": response_body.get("content", []), "msg_id": msg_id})
            print(f"[RUN] Assistant thought: {thought}")

            if stop_reason == 'tool_use':
                print(f"[RUN] Assistant action: Call tool `{action}`, parameters: {params}")
                # self.tool_call_counter += 1
                
                result = self._execute_tool(action, params)
                self.ctx_counter += 1
                msg_id_tool = self.ctx_counter
                self.full_history.append({
                    "role": "tool", 
                    "content": result, 
                    "msg_id": msg_id_tool,
                    "msg_id(invoking_assistant)": msg_id,
                    "tool_use_id": tool_use_id,
                    "tool_name": action
                })
                
                result_preview = json.dumps(result, ensure_ascii=False)
                if len(result_preview) > 200:
                    result_preview = result_preview[:200] + "..."
                print(f"[RUN] Tool result (ID: {msg_id_tool}): {result_preview}")

                if action == "finish":
                    print(f"\n--- Final Answer --- \n{result.get('final_answer', 'No final answer provided.')}")
                    break
            
            elif stop_reason == 'end_turn':
                print("[INFO] Model finished thinking but did not call tool, process ended.")
                if thought: print(f"\n--- Final Answer --- \n{thought}")
                break
            
            else:
                print(f"[INFO] Process terminated due to stop_reason '{stop_reason}'.")
                break

            turn += 1
        
        if turn > self.max_turns:
            print(f"[INFO] Reached max rounds {self.max_turns}, stopping execution.")
        
        return self._build_api_payload() # full api payload for logging
