from typing import (
    List,
    Optional,
    Tuple,
)

from src.retriever import BaseRetriever
from src.schema import Document
from src.service.generation_stage_service.question_answering_service import QuestionAnsweringService
from src.service.rag_service.base_rag_service import BaseRAGService
from src.service.retrieval_stage_service.memory_reranking_service import MemoryRerankingService
from src.service.retrieval_stage_service.sub_query_generation_service import SubQueryGenerationService


class AdvancedRAGService(BaseRAGService):
    def __init__(
        self,
        retriever: BaseRetriever,
        sub_query_generation_service: SubQueryGenerationService,
        memory_reranking_service: MemoryRerankingService,
        question_answering_service: QuestionAnsweringService,
    ) -> None:
        self.retriever = retriever
        self.sub_query_generation_service = sub_query_generation_service
        self.memory_reranking_service = memory_reranking_service
        self.question_answering_service = question_answering_service

    def _retrieve_for_sub_queries(
        self,
        collection_name,
        sub_queries: List[str],
    ) -> Tuple[List[str], List[str], List[float]]:
        ret_items = []
        ret_item_ids = []
        ret_item_scores = []
        for sub_query in sub_queries:
            items, item_ids, item_scores = self.retriever.retrieve(
                collection_name=collection_name,
                query=sub_query,
            )
            ret_items.extend(items)
            ret_item_ids.extend(item_ids)
            ret_item_scores.extend(item_scores)
        return ret_items, ret_item_ids, ret_item_scores

    def retrieve(
        self,
        collection_name: str,
        query: str,
        documents: List[Document],
        answer_element_universe: Optional[List[str]] = None,
    ) -> Tuple[List[str], List[str], List[float]]:
        sub_queries = self.sub_query_generation_service.generate_sub_queries(
            query=query,
            documents=documents,
            answer_element_universe=answer_element_universe,
        )
        ret_memory_contents, ret_memory_ids, ret_memory_scores = self._retrieve_for_sub_queries(
            collection_name=collection_name,
            sub_queries=sub_queries,
        )
        ret_memory_contents, ret_memory_ids, ret_memory_scores = self.memory_reranking_service.rerank(
            ret_memory_contents=ret_memory_contents,
            ret_memory_ids=ret_memory_ids,
            ret_memory_scores=ret_memory_scores,
        )
        return ret_memory_contents, ret_memory_ids, ret_memory_scores

    def generate(
        self,
        query: str,
        documents: List[Document],
        ret_memory_contents: List[str],
        answer_element_universe: Optional[List[str]] = None,
    ) -> str:
        raw_answer = self.question_answering_service.generate_answer(
            query=query,
            documents=documents,
            ret_memory_contents=ret_memory_contents,
            answer_element_universe=answer_element_universe,
        )
        return raw_answer

    def forward(
        self,
        collection_name: str,
        query: str,
        documents: List[Document],
        answer_element_universe: Optional[List[str]] = None,
    ) -> Tuple[str, List[str], List[str], List[float]]:
        ret_memory_contents, ret_memory_ids, ret_memory_scores = self.retrieve(
            collection_name=collection_name,
            query=query,
            documents=documents,
            answer_element_universe=answer_element_universe,
        )
        raw_answer = self.generate(
            query=query,
            documents=documents,
            ret_memory_contents=ret_memory_contents,
            answer_element_universe=answer_element_universe,
        )
        return raw_answer, ret_memory_contents, ret_memory_ids, ret_memory_scores
