from abc import abstractmethod

import math
import numpy as np
import torch as th
import torch.nn as nn
import torch.nn.functional as F

from .fp16_util import convert_module_to_f16, convert_module_to_f32
from .nn import (
    checkpoint,
    conv_nd,
    linear,
    avg_pool_nd,
    zero_module,
    normalization,
    timestep_embedding,
)
from .prune_util import pruned_group_norm, get_new_out_channel


class AttentionPool2d(nn.Module):
    """
    Adapted from CLIP: https://github.com/openai/CLIP/blob/main/clip/model.py
    """

    def __init__(
        self,
        spacial_dim: int,
        embed_dim: int,
        num_heads_channels: int,
        output_dim: int = None,
    ):
        super().__init__()
        self.positional_embedding = nn.Parameter(
            th.randn(embed_dim, spacial_dim ** 2 + 1) / embed_dim ** 0.5
        )
        self.qkv_proj = conv_nd(1, embed_dim, 3 * embed_dim, 1)
        self.c_proj = conv_nd(1, embed_dim, output_dim or embed_dim, 1)
        self.num_heads = embed_dim // num_heads_channels
        self.attention = QKVAttention(self.num_heads)

    def forward(self, x):
        b, c, *_spatial = x.shape
        x = x.reshape(b, c, -1)  # NC(HW)
        x = th.cat([x.mean(dim=-1, keepdim=True), x], dim=-1)  # NC(HW+1)
        x = x + self.positional_embedding[None, :, :].to(x.dtype)  # NC(HW+1)
        x = self.qkv_proj(x)
        x = self.attention(x)
        x = self.c_proj(x)
        return x[:, :, 0]


class TimestepBlock(nn.Module):
    """
    Any module where forward() takes timestep embeddings as a second argument.
    """

    @abstractmethod
    def forward(self, x, emb):
        """
        Apply the module to `x` given `emb` timestep embeddings.
        """


class TimestepEmbedSequential(nn.Sequential, TimestepBlock):
    """
    A sequential module that passes timestep embeddings to the children that
    support it as an extra input.
    """

    def forward(self, x, emb):
        for layer in self:
            if isinstance(layer, TimestepBlock):
                x = layer(x, emb)
            else:
                x = layer(x)
        return x


class Upsample(nn.Module):
    """
    An upsampling layer with an optional convolution.

    :param channels: channels in the inputs and outputs.
    :param use_conv: a bool determining if a convolution is applied.
    :param dims: determines if the signal is 1D, 2D, or 3D. If 3D, then
                 upsampling occurs in the inner-two dimensions.
    """

    def __init__(self, channels, use_conv, dims=2, out_channels=None):
        super().__init__()
        self.channels = channels
        self.out_channels = out_channels or channels
        self.use_conv = use_conv
        self.dims = dims
        if use_conv:
            self.conv = conv_nd(dims, self.channels, self.out_channels, 3, padding=1)

    def forward(self, x):
        if self.dims == 3:
            x = F.interpolate(
                x, (x.shape[2], x.shape[3] * 2, x.shape[4] * 2), mode="nearest"
            )
        else:
            x = F.interpolate(x, scale_factor=2, mode="nearest")
        if self.use_conv:
            x = self.conv(x)
        return x


class Downsample(nn.Module):
    """
    A downsampling layer with an optional convolution.

    :param channels: channels in the inputs and outputs.
    :param use_conv: a bool determining if a convolution is applied.
    :param dims: determines if the signal is 1D, 2D, or 3D. If 3D, then
                 downsampling occurs in the inner-two dimensions.
    """

    def __init__(self, channels, use_conv, dims=2, out_channels=None):
        super().__init__()
        self.channels = channels
        self.out_channels = out_channels or channels
        self.use_conv = use_conv
        self.dims = dims
        stride = 2 if dims != 3 else (1, 2, 2)
        if use_conv:
            self.op = conv_nd(
                dims, self.channels, self.out_channels, 3, stride=stride, padding=1
            )
        else:
            assert self.channels == self.out_channels
            self.op = avg_pool_nd(dims, kernel_size=stride, stride=stride)

    def forward(self, x):
        return self.op(x)


class ResBlock(TimestepBlock):
    """
    A residual block that can optionally change the number of channels.

    :param channels: the number of input channels.
    :param emb_channels: the number of timestep embedding channels.
    :param dropout: the rate of dropout.
    :param out_channels: if specified, the number of out channels.
    :param use_conv: if True and out_channels is specified, use a spatial
        convolution instead of a smaller 1x1 convolution to change the
        channels in the skip connection.
    :param dims: determines if the signal is 1D, 2D, or 3D.
    :param use_checkpoint: if True, use gradient checkpointing on this module.
    :param up: if True, use this block for upsampling.
    :param down: if True, use this block for downsampling.
    """

    def __init__(
        self,
        channels,
        emb_channels,
        dropout,
        out_channels=None,
        use_conv=False,
        use_scale_shift_norm=False,
        dims=2,
        use_checkpoint=False,
        up=False,
        down=False,
        mid_name='input_blocks.1.0.', #####
        reverse_info=None, #####
        train_mode=False, #####
    ):
        super().__init__()
        self.channels = channels
        self.emb_channels = emb_channels
        self.dropout = dropout
        self.out_channels = out_channels or channels
        self.use_conv = use_conv
        self.use_checkpoint = use_checkpoint
        self.use_scale_shift_norm = use_scale_shift_norm

        #####
        self.train_mode = train_mode
        name = mid_name+'in_layers.2'
        self.name = name
        self.layer_name = name
        pruned_in_channels = len(reverse_info[name][0])
        self.input_resblock_ch = reverse_info[name][0]
        pruned_out_channels = len(reverse_info[name][1])
        self.embed_reversed_c = reverse_info[name][1]


        self.in_layers = nn.Sequential(
            normalization(channels),
            nn.SiLU(),
            conv_nd(dims, pruned_in_channels, pruned_out_channels, 3, padding=1),
        )
        #####

        self.updown = up or down

        if up:
            self.h_upd = Upsample(channels, False, dims)
            self.x_upd = Upsample(channels, False, dims)
        elif down:
            self.h_upd = Downsample(channels, False, dims)
            self.x_upd = Downsample(channels, False, dims)
        else:
            self.h_upd = self.x_upd = nn.Identity()

        #####
        name_emb = mid_name+'emb_layers.1'
        self.emb_in_ch = reverse_info[name_emb][0]
        self.emb_out_ch = reverse_info[name_emb][1]

        self.emb_layers = nn.Sequential(
            nn.SiLU(),
            linear(
                len(self.emb_in_ch),
                len(self.emb_out_ch),
            ),
        )
        #####

        #####
        name = mid_name+'out_layers.3'
        pruned_out_channels = len(reverse_info[name][1])
        self.skip_reversed_c = reverse_info[name][1]
        self.h_channels = reverse_info[name][1]

        self.out_layers = nn.Sequential(
            normalization(self.out_channels),
            nn.SiLU(),
            nn.Dropout(p=dropout),
            zero_module(
                conv_nd(dims, len(self.embed_reversed_c), pruned_out_channels, 3, padding=1)
            ),
        )
        #####

        if self.out_channels == channels:
            #####
            self.skpi_is_conv = False
            #####
            self.skip_connection = nn.Identity()
        elif use_conv:
            #####
            self.skpi_is_conv = True
            name = mid_name+'skip_connection'
            pruned_in_channels = len(reverse_info[name][0])
            pruned_out_channels = len(reverse_info[name][1])
            self.skip_connection = conv_nd(
                dims, pruned_in_channels, pruned_out_channels, 3, padding=1
            )
            self.skip_reversed_c = reverse_info[name][1]
            #####
        else:
            #####
            self.skpi_is_conv = True
            name = mid_name+'skip_connection'
            pruned_in_channels = len(reverse_info[name][0])
            pruned_out_channels = len(reverse_info[name][1])
            self.skip_connection = conv_nd(dims, pruned_in_channels, pruned_out_channels, 1)
            self.skip_reversed_c = reverse_info[name][1]
            #####

    def forward(self, x, emb):
        """
        Apply the block to a Tensor, conditioned on a timestep embedding.

        :param x: an [N x C x ...] Tensor of features.
        :param emb: an [N x emb_channels] Tensor of timestep embeddings.
        :return: an [N x C x ...] Tensor of outputs.
        """
        return checkpoint(
            self._forward, (x, emb), self.parameters(), self.use_checkpoint
        )

    def _forward(self, x, emb):

        if self.updown:
            #####
            in_gn, in_rest, in_conv = self.in_layers[0], self.in_layers[1:-1], self.in_layers[-1]
            x_board_shape = (x.shape[0], self.channels, x.shape[2], x.shape[3])
            x_board = th.zeros(x_board_shape, dtype=x.dtype, device=x.device)
            x_board[:,self.input_resblock_ch] = x
            h = in_gn(x_board)
            #####

            h = in_rest(h)
            h = self.h_upd(h)
            x = self.x_upd(x)

            # @@@@@@@@
            h = h[:,self.input_resblock_ch]

            h = in_conv(h)

        else:
            #####
            in_gn, in_res = self.in_layers[0], self.in_layers[1:]
            x_board_shape = (x.shape[0], self.channels, x.shape[2], x.shape[3])
            x_board = th.zeros(x_board_shape, dtype=x.dtype, device=x.device)

            x_board[:,self.input_resblock_ch] = x

            h = in_gn(x_board)

            # @@@@@@@@
            h = h[:,self.input_resblock_ch]

            h = in_res(h)
            #####

        #####
        emb_out = self.emb_layers(emb[:,self.emb_in_ch]).type(h.dtype)
        assert self.use_scale_shift_norm == True
        emb_out_shape = (emb_out.shape[0], 2 * self.out_channels)
        emb_out_board = th.zeros(emb_out_shape, dtype=emb_out.dtype, device=emb_out.device)
        emb_out_board[:,self.emb_out_ch] = emb_out
        #####
        while len(emb_out_board.shape) < len(h.shape):
            emb_out_board = emb_out_board[..., None]
        if self.use_scale_shift_norm:
            out_norm, out_rest = self.out_layers[0], self.out_layers[1:]
            scale, shift = th.chunk(emb_out_board, 2, dim=1)

            #####
            h_board_shape = (h.shape[0], scale.shape[1], h.shape[2], h.shape[3])
            h_board = th.zeros(h_board_shape, dtype=h.dtype, device = h.device)
            h_board[:,self.embed_reversed_c] = h
            h = out_norm(h_board) * (1 + scale) + shift
            #####

            # @@@@@@@@
            h = h[:,self.embed_reversed_c]

            h = out_rest(h)
        else:
            h = h + emb_out
            h = self.out_layers(h)


        #####
        skip_out = self.skip_connection(x)
        if h.shape[1] != skip_out.shape[1]:
            #if self.train_mode:
            return_out = th.zeros(skip_out.shape, dtype=h.dtype, device=h.device)
            return_out[:,self.h_channels] = h
            return_out += skip_out
        else:    
            return_out = h + skip_out
        #####

        return return_out


class AttentionBlock(nn.Module):
    """
    An attention block that allows spatial positions to attend to each other.

    Originally ported from here, but adapted to the N-d case.
    https://github.com/hojonathanho/diffusion/blob/1e0dceb3b3495bbe19116a5e1b3596cd0706c543/diffusion_tf/models/unet.py#L66.
    """

    def __init__(
        self,
        channels,
        num_heads=1,
        num_head_channels=-1,
        use_checkpoint=False,
        use_new_attention_order=False,
        att_in_ch=None,
        mid_name='input_blocks.1.1.', #####
        reverse_info=None, #####
        train_mode=False, #####
    ):
        super().__init__()
        self.channels = channels
        if num_head_channels == -1:
            self.num_heads = num_heads
        else:
            assert (
                channels % num_head_channels == 0
            ), f"q,k,v channels {channels} is not divisible by num_head_channels {num_head_channels}"
            self.num_heads = channels // num_head_channels
        self.use_checkpoint = use_checkpoint

        ######
        self.norm = normalization(channels)
        self.att_in_ch = att_in_ch
        self.train_mode = train_mode
        #####

        #####
        name_qkv = mid_name + 'qkv'
        self.qkv_in_ch = reverse_info[name_qkv][0]
        self.qkv_out_ch = reverse_info[name_qkv][1]
        self.qkv = conv_nd(1, len(self.qkv_in_ch), len(self.qkv_out_ch), 1)
        #####

        if use_new_attention_order:
            # split qkv before split heads
            self.attention = QKVAttention(self.num_heads)
        else:
            # split heads before split qkv
            self.attention = QKVAttentionLegacy(self.num_heads)

        #####
        name_po = mid_name + 'proj_out'
        self.po_in_ch = reverse_info[name_po][0]
        self.po_out_ch = reverse_info[name_po][1]
        self.proj_out = zero_module(conv_nd(1, len(self.po_in_ch), len(self.po_out_ch), 1))
        #####

    def forward(self, x):
        return checkpoint(self._forward, (x,), self.parameters(), True)

    def _forward(self, x):
        b, c, *spatial = x.shape
        x = x.reshape(b, c, -1)

        #####
        x_board_shape = (x.shape[0], self.channels, x.shape[2])
        x_board = th.zeros(x_board_shape, dtype=x.dtype, device=x.device)
        x_board[:,self.att_in_ch] = x
        x1 = self.norm(x_board)
        #####

        #####
        qkv_shape = (x1.shape[0], self.channels*3, x1.shape[2])
        qkv = self.qkv(x1[:,self.qkv_in_ch])
        qkv_board = th.zeros(qkv_shape, dtype=x1.dtype, device=x1.device)
        qkv_board[:,self.qkv_out_ch] = qkv

        h = self.attention(qkv_board)

        h = self.proj_out(h[:,self.po_in_ch])

        #if self.train_mode:
        att_out = th.zeros(x_board.shape, dtype=x_board.dtype, device=x_board.device)
        att_out[:,self.po_out_ch] = h
        att_out += x_board
        bo, co, *spatialo = x_board.shape
        #####

        att_out = att_out.reshape(bo, co, *spatial)

        return att_out


def count_flops_attn(model, _x, y):
    """
    A counter for the `thop` package to count the operations in an
    attention operation.
    Meant to be used like:
        macs, params = thop.profile(
            model,
            inputs=(inputs, timestamps),
            custom_ops={QKVAttention: QKVAttention.count_flops},
        )
    """
    b, c, *spatial = y[0].shape
    num_spatial = int(np.prod(spatial))
    # We perform two matmuls with the same number of ops.
    # The first computes the weight matrix, the second computes
    # the combination of the value vectors.
    matmul_ops = 2 * b * (num_spatial ** 2) * c
    model.total_ops += th.DoubleTensor([matmul_ops])


class QKVAttentionLegacy(nn.Module):
    """
    A module which performs QKV attention. Matches legacy QKVAttention + input/ouput heads shaping
    """

    def __init__(self, n_heads):
        super().__init__()
        self.n_heads = n_heads

    def forward(self, qkv):
        """
        Apply QKV attention.

        :param qkv: an [N x (H * 3 * C) x T] tensor of Qs, Ks, and Vs.
        :return: an [N x (H * C) x T] tensor after attention.
        """
        bs, width, length = qkv.shape
        assert width % (3 * self.n_heads) == 0
        ch = width // (3 * self.n_heads)
        q, k, v = qkv.reshape(bs * self.n_heads, ch * 3, length).split(ch, dim=1)
        scale = 1 / math.sqrt(math.sqrt(ch))
        weight = th.einsum(
            "bct,bcs->bts", q * scale, k * scale
        )  # More stable with f16 than dividing afterwards
        weight = th.softmax(weight.float(), dim=-1).type(weight.dtype)
        a = th.einsum("bts,bcs->bct", weight, v)
        return a.reshape(bs, -1, length)

    @staticmethod
    def count_flops(model, _x, y):
        return count_flops_attn(model, _x, y)


class QKVAttention(nn.Module):
    """
    A module which performs QKV attention and splits in a different order.
    """

    def __init__(self, n_heads):
        super().__init__()
        self.n_heads = n_heads

    def forward(self, qkv):
        """
        Apply QKV attention.

        :param qkv: an [N x (3 * H * C) x T] tensor of Qs, Ks, and Vs.
        :return: an [N x (H * C) x T] tensor after attention.
        """
        bs, width, length = qkv.shape
        assert width % (3 * self.n_heads) == 0
        ch = width // (3 * self.n_heads)
        q, k, v = qkv.chunk(3, dim=1)
        scale = 1 / math.sqrt(math.sqrt(ch))
        weight = th.einsum(
            "bct,bcs->bts",
            (q * scale).view(bs * self.n_heads, ch, length),
            (k * scale).view(bs * self.n_heads, ch, length),
        )  # More stable with f16 than dividing afterwards
        weight = th.softmax(weight.float(), dim=-1).type(weight.dtype)
        a = th.einsum("bts,bcs->bct", weight, v.reshape(bs * self.n_heads, ch, length))
        return a.reshape(bs, -1, length)

    @staticmethod
    def count_flops(model, _x, y):
        return count_flops_attn(model, _x, y)


class P_UNetModel(nn.Module):
    """
    The full UNet model with attention and timestep embedding.

    :param in_channels: channels in the input Tensor.
    :param model_channels: base channel count for the model.
    :param out_channels: channels in the output Tensor.
    :param num_res_blocks: number of residual blocks per downsample.
    :param attention_resolutions: a collection of downsample rates at which
        attention will take place. May be a set, list, or tuple.
        For example, if this contains 4, then at 4x downsampling, attention
        will be used.
    :param dropout: the dropout probability.
    :param channel_mult: channel multiplier for each level of the UNet.
    :param conv_resample: if True, use learned convolutions for upsampling and
        downsampling.
    :param dims: determines if the signal is 1D, 2D, or 3D.
    :param num_classes: if specified (as an int), then this model will be
        class-conditional with `num_classes` classes.
    :param use_checkpoint: use gradient checkpointing to reduce memory usage.
    :param num_heads: the number of attention heads in each attention layer.
    :param num_heads_channels: if specified, ignore num_heads and instead use
                               a fixed channel width per attention head.
    :param num_heads_upsample: works with num_heads to set a different number
                               of heads for upsampling. Deprecated.
    :param use_scale_shift_norm: use a FiLM-like conditioning mechanism.
    :param resblock_updown: use residual blocks for up/downsampling.
    :param use_new_attention_order: use a different attention pattern for potentially
                                    increased efficiency.
    """

    def __init__(
        self,
        image_size,
        in_channels,
        model_channels,
        out_channels,
        num_res_blocks,
        attention_resolutions,
        dropout=0,
        channel_mult=(1, 2, 4, 8),
        conv_resample=True,
        dims=2,
        num_classes=None,
        use_checkpoint=False,
        use_fp16=False,
        num_heads=1,
        num_head_channels=-1,
        num_heads_upsample=-1,
        use_scale_shift_norm=False,
        resblock_updown=False,
        use_new_attention_order=False,
        reverse_info=None,
        train_mode=False,
    ):
        super().__init__()

        if num_heads_upsample == -1:
            num_heads_upsample = num_heads

        self.image_size = image_size
        self.in_channels = in_channels
        self.model_channels = model_channels
        self.out_channels = out_channels
        self.num_res_blocks = num_res_blocks
        self.attention_resolutions = attention_resolutions
        self.dropout = dropout
        self.channel_mult = channel_mult
        self.conv_resample = conv_resample
        self.num_classes = num_classes
        self.use_checkpoint = use_checkpoint
        self.dtype = th.float16 if use_fp16 else th.float32
        self.num_heads = num_heads
        self.num_head_channels = num_head_channels
        self.num_heads_upsample = num_heads_upsample

        #####
        self.reverse_info = reverse_info
        self.train_mode = train_mode
        #####

        time_embed_dim = model_channels * 4

        #####
        lin_0_out_ch = reverse_info['time_embed.0'][1]
        lin_1_in_ch = reverse_info['time_embed.2'][0]
        self.time_embed = nn.Sequential(
            linear(model_channels, len(lin_0_out_ch)),
            nn.SiLU(),
            linear(len(lin_1_in_ch), time_embed_dim),
        )
        #####

        if self.num_classes is not None:
            self.label_emb = nn.Embedding(num_classes, time_embed_dim)

        ch = input_ch = int(channel_mult[0] * model_channels)
        
        #####
        name = 'input_blocks.'+str(0)+'.'+str(0)
        pruned_ch = len(reverse_info[name][1])
        self.input_blocks = nn.ModuleList(
            [TimestepEmbedSequential(conv_nd(dims, in_channels, pruned_ch, 3, padding=1))]
        )
        #####

        #####
        base_name = 'input_blocks.'
        num_input_blocks = 1
        #####
        self._feature_size = ch
        input_block_chans = [ch]
        ds = 1
        for level, mult in enumerate(channel_mult):
            for _ in range(num_res_blocks):

                #####
                mid_name = base_name+str(num_input_blocks)+'.0.'
                #####

                layers = [
                    ResBlock(
                        ch,
                        time_embed_dim,
                        dropout,
                        out_channels=int(mult * model_channels),
                        dims=dims,
                        use_checkpoint=use_checkpoint,
                        use_scale_shift_norm=use_scale_shift_norm,
                        mid_name=mid_name, ######
                        reverse_info=reverse_info, #####
                        train_mode=self.train_mode, #####
                    )
                ]
                ch = int(mult * model_channels)
                if ds in attention_resolutions:
                    #####
                    name1 = base_name+str(num_input_blocks)+'.0.out_layers.3'
                    name2 = base_name+str(num_input_blocks)+'.0.skip_connection'
                    name3 = base_name+str(num_input_blocks)+'.0.in_layers.2'
                    if name2 in reverse_info.keys():
                        _, att_in_ch = get_new_out_channel(reverse_info[name1][1],reverse_info[name2][1])
                    else:
                        _, att_in_ch = get_new_out_channel(reverse_info[name1][1],reverse_info[name3][0])
                    mid_name = base_name+str(num_input_blocks)+'.1.'
                    #####
                    layers.append(
                        AttentionBlock(
                            ch,
                            use_checkpoint=use_checkpoint,
                            num_heads=num_heads,
                            num_head_channels=num_head_channels,
                            use_new_attention_order=use_new_attention_order,
                            att_in_ch=att_in_ch, #####
                            mid_name=mid_name, ######
                            reverse_info=reverse_info, #####
                            train_mode=self.train_mode, #####
                        )
                    )
                self.input_blocks.append(TimestepEmbedSequential(*layers))
                #####
                num_input_blocks += 1
                #####
                self._feature_size += ch
                input_block_chans.append(ch)
            if level != len(channel_mult) - 1:
                out_ch = ch
                #####
                mid_name = base_name+str(num_input_blocks)+'.0.'
                #####
                self.input_blocks.append(
                    TimestepEmbedSequential(
                        ResBlock(
                            ch,
                            time_embed_dim,
                            dropout,
                            out_channels=out_ch,
                            dims=dims,
                            use_checkpoint=use_checkpoint,
                            use_scale_shift_norm=use_scale_shift_norm,
                            down=True,
                            mid_name=mid_name, ######
                            reverse_info=reverse_info, #####
                            train_mode=self.train_mode, #####
                        )
                        if resblock_updown
                        else Downsample(
                            ch, conv_resample, dims=dims, out_channels=out_ch
                        )
                    )
                )
                #####
                num_input_blocks += 1
                #####
                ch = out_ch
                input_block_chans.append(ch)
                ds *= 2
                self._feature_size += ch


        #####
        name1 = 'middle_block.0.out_layers.3'
        name2 = 'middle_block.0.skip_connection'
        name3 = 'middle_block.0.in_layers.2'
        if name2 in reverse_info.keys():
            _, att_in_ch = get_new_out_channel(reverse_info[name1][1],reverse_info[name2][1])
        else:
            _, att_in_ch = get_new_out_channel(reverse_info[name1][1],reverse_info[name3][0])
        #####

        self.middle_block = TimestepEmbedSequential(
            ResBlock(
                ch,
                time_embed_dim,
                dropout,
                dims=dims,
                use_checkpoint=use_checkpoint,
                use_scale_shift_norm=use_scale_shift_norm,
                mid_name='middle_block.0.', #####
                reverse_info=reverse_info, #####
                train_mode=self.train_mode, #####
            ),
            AttentionBlock(
                ch,
                use_checkpoint=use_checkpoint,
                num_heads=num_heads,
                num_head_channels=num_head_channels,
                use_new_attention_order=use_new_attention_order,
                att_in_ch=att_in_ch, #####
                mid_name='middle_block.1.', #####
                reverse_info=reverse_info, #####
                train_mode=self.train_mode, #####
            ),
            ResBlock(
                ch,
                time_embed_dim,
                dropout,
                dims=dims,
                use_checkpoint=use_checkpoint,
                use_scale_shift_norm=use_scale_shift_norm,
                mid_name='middle_block.2.', #####
                reverse_info=reverse_info, #####
                train_mode=self.train_mode, #####
            ),
        )
        self._feature_size += ch

        #####
        base_name = 'output_blocks.'
        num_output_blocks = 0
        #####

        self.output_blocks = nn.ModuleList([])
        for level, mult in list(enumerate(channel_mult))[::-1]:
            for i in range(num_res_blocks + 1):
                #####
                have_att = False
                #####

                ich = input_block_chans.pop()

                #####
                mid_name = base_name+str(num_output_blocks)+'.0.'
                #####
                layers = [
                    ResBlock(
                        ch + ich,
                        time_embed_dim,
                        dropout,
                        out_channels=int(model_channels * mult),
                        dims=dims,
                        use_checkpoint=use_checkpoint,
                        use_scale_shift_norm=use_scale_shift_norm,
                        mid_name=mid_name, #####
                        reverse_info=reverse_info, #####
                        train_mode=self.train_mode, #####
                    )
                ]
                ch = int(model_channels * mult)
                if ds in attention_resolutions:
                    #####
                    have_att = True
                    name1 = base_name+str(num_output_blocks)+'.0.out_layers.3'
                    name2 = base_name+str(num_output_blocks)+'.0.skip_connection'
                    name3 = base_name+str(num_input_blocks)+'.0.in_layers.2'
                    if name2 in reverse_info.keys():
                        _, att_in_ch = get_new_out_channel(reverse_info[name1][1],reverse_info[name2][1])
                    else:
                        _, att_in_ch = get_new_out_channel(reverse_info[name1][1],reverse_info[name3][0])
                    mid_name = base_name+str(num_output_blocks)+'.1.'
                    #####
                    layers.append(
                        AttentionBlock(
                            ch,
                            use_checkpoint=use_checkpoint,
                            num_heads=num_heads_upsample,
                            num_head_channels=num_head_channels,
                            use_new_attention_order=use_new_attention_order,
                            att_in_ch=att_in_ch, #####
                            mid_name=mid_name, #####
                            reverse_info=reverse_info, #####
                            train_mode=self.train_mode, #####
                        )
                    )
                if level and i == num_res_blocks:
                    out_ch = ch

                    #####
                    if have_att:
                        mid_name = base_name+str(num_output_blocks)+'.2.'
                    else:
                        mid_name = base_name+str(num_output_blocks)+'.1.'
                    #####

                    layers.append(
                        ResBlock(
                            ch,
                            time_embed_dim,
                            dropout,
                            out_channels=out_ch,
                            dims=dims,
                            use_checkpoint=use_checkpoint,
                            use_scale_shift_norm=use_scale_shift_norm,
                            up=True,
                            mid_name=mid_name, #####
                            reverse_info=reverse_info, #####
                            train_mode=self.train_mode, #####
                        )
                        if resblock_updown
                        else Upsample(ch, conv_resample, dims=dims, out_channels=out_ch)
                    )
                    ds //= 2
                self.output_blocks.append(TimestepEmbedSequential(*layers))
                #####
                num_output_blocks += 1
                #####
                self._feature_size += ch

        ######
        name1 = mid_name + 'out_layers.3'
        name2 = mid_name + 'skip_connection'
        name3 = mid_name + 'in_layers.2'
        if name2 in reverse_info.keys():
            the_last_ch, self.out_inch = get_new_out_channel(reverse_info[name1][1],reverse_info[name2][1])
        else:
            the_last_ch, self.out_inch = get_new_out_channel(reverse_info[name1][1],reverse_info[name3][0])

        self.out_norm_ch = ch
        #####
        self.out = nn.Sequential(
            normalization(ch),
            nn.SiLU(),
            zero_module(conv_nd(dims, the_last_ch, out_channels, 3, padding=1)),
        )

    def convert_to_fp16(self):
        """
        Convert the torso of the model to float16.
        """
        self.input_blocks.apply(convert_module_to_f16)
        self.middle_block.apply(convert_module_to_f16)
        self.output_blocks.apply(convert_module_to_f16)

    def convert_to_fp32(self):
        """
        Convert the torso of the model to float32.
        """
        self.input_blocks.apply(convert_module_to_f32)
        self.middle_block.apply(convert_module_to_f32)
        self.output_blocks.apply(convert_module_to_f32)

    def forward(self, x, timesteps, y=None):
        """
        Apply the model to an input batch.

        :param x: an [N x C x ...] Tensor of inputs.
        :param timesteps: a 1-D batch of timesteps.
        :param y: an [N] Tensor of labels, if class-conditional.
        :return: an [N x C x ...] Tensor of outputs.
        """
        assert (y is not None) == (
            self.num_classes is not None
        ), "must specify y if and only if the model is class-conditional"

        hs = []
        emb = self.time_embed(timestep_embedding(timesteps, self.model_channels))

        if self.num_classes is not None:
            assert y.shape == (x.shape[0],)
            emb = emb + self.label_emb(y)

        h = x.type(self.dtype)

        for module in self.input_blocks:
            h = module(h, emb)
            hs.append(h)

        h = self.middle_block(h, emb)

        for module in self.output_blocks:
            h = th.cat([h, hs.pop()], dim=1)
            h = module(h, emb)

        h = h.type(th.cuda.FloatTensor)

        #####
        out_norm, out_res = self.out[0], self.out[1:]
        h_board_shape = (h.shape[0], self.out_norm_ch, h.shape[2], h.shape[3])
        h_board = th.zeros(h_board_shape, dtype=h.dtype, device=h.device)
        h_board[:,self.out_inch] = h
        h1 = out_norm(h_board)

        # @@@@@@@@
        h1 = h1[:,self.out_inch]

        out = out_res(h1)
        out = out.type(x.dtype)
        return out
