'''
- ranking.py
- This file handles interfacing crucial ranking functions with the ranking utilities
'''

# External imports

# Internal imports
from traceback import print_tb
from src.core.configuration.embedding_conf import *
import src.core.interface.similarity as similarity
from src.utils.ranking import fasttext, glove, word2vec, bert, use

# Ranking update distribution (These should add up to one, ideally)
PREV_WEIGHT = 0.9
UPDATED_WEIGHT = 0.1




'''
----------load_embedding----------
- This function loads in the current feature vector set to work with
-----Inputs-----
- embedding - the name of the embedding to use
- schema - the currently-active schema
-----Output-----
- embedding - the dictionary of embeddings
'''
def load_embedding(embedding, schema='flight-delay'):
    if "fasttext" in embedding:
        if "finetuned" in embedding:
            return fasttext.load_finetuned_embedding(FASTTEXT_TUNED_LOC)
        else:
            return fasttext.load_embedding(FASTTEXT_LOC)
    elif "glove" in embedding:
        if "finetuned" in embedding:
            pass
        else:
            return glove.load_embedding(GLOVE_LOC)
    elif "word2vec" in embedding:
        if "finetuned" in embedding:
            pass
        else:
            return word2vec.load_embedding(WORD2VEC_LOC)
    elif "bert" in embedding:
        if "finetuned" in embedding:
            pass
        else:
            return bert.load_embedding(schema)
    elif "roberta" in embedding:
        if "finetuned" in embedding:
            pass
        else:
            pass
    elif "xlnet" in embedding:
        if "finetuned" in embedding:
            pass
        else:
            pass
    elif "use" in embedding:
        return use.load_embedding(USE_LOC)
    print("No embedding detected")
    return


'''
----------update_rankings----------
- This function calls the current ranking implementation
-----Inputs-----
- text - the user input to parse
- features - the features to rank
- ranking - the ranking to update
- embedding - the dictionary of embeddings
- configuration - the currently-active configuration
- normalization - the normalization technique to use (defaults to softmax)
- < active_feature - the currently-active feature >-----Not included - I don't think we'll need this, but keeping just in case
-----Output-----
- rankings - the resultant ranked list of items
'''
def update_rankings(text, features, ranking, embedding, configuration, normalization="softmax"):
    # Get the proper ranked list based on distances
    updated_ranking = similarity.rank_values(text, features, embedding, configuration)
    # Normalize the ranked list, and generate a ranked list of probabilities
    normalized_ranking = similarity.normalize(updated_ranking, normalization)

    # Update the current distribution based on this new ranking
    # (Note: this can probably be more efficient than it currently is, but the input size is sufficiently small enough that it shouldn't matter)
    refined_ranking = []
    for value in ranking:
        updated_dist = [item for item in normalized_ranking if item["name"] == value["name"]][0]
        refined_ranking.append({"name":value["name"],"distribution":(value["distribution"] * PREV_WEIGHT) + (updated_dist["distribution"] * UPDATED_WEIGHT)})

    # Return the sorted ranked list of probabilities
    return similarity.normalize(refined_ranking, normalization)



'''
----------update_joint_ranking----------
- This function is a concept for the joint ranking implementation
-----Inputs-----
- phrases - the phrases [{"text":str, "confidence":float}]
  - text - the user input to parse
  - confidence - the probability that the sub-phrase is the target phrase (obtained from NER)
- features - the features to rank
- ranking - the ranking to update
- embedding - the dictionary of embeddings
- configuration - the currently-active configuration
- normalization - the normalization technique to use (defaults to softmax)
-----Output-----
- rankings - the resultant ranked list of items
'''
def update_joint_ranking(phrases, features, ranking, embedding, configuration, normalization="softmax"):
    # Initialize the joint ranking with a uniform 0 probability
    # print("inside joint ranking")
    # print(ranking)
    joint_ranking = []
    for value in ranking:
        joint_ranking.append({"name":value["name"],"distribution":0})
    # Determine the joint probability for each phrase, and slot the maximum into the list
    for phrase in phrases:
        # Get P(Y=1|q) from the NER technique (confidence parameter)
        # Get P(x|q,Y=1) from the embedding
        updated_ranking = similarity.rank_values(phrase["text"], features, embedding, configuration)
        # Normalize the resultant ranked list, and generate a ranked list of probabilities
        # for i in range(len(updated_ranking)):
        #     if i < len(updated_ranking)/2:
        #         divisor = max(1, 100/(i+1))
        #         updated_ranking[i]["distribution"] = updated_ranking[i]["distribution"] * (divisor)
        
        # print(updated_ranking)
        # normalized_ranking = similarity.normalize(updated_ranking, normalization)# May not be necessary, test and see if I can remove (Initial hunch is don't remove, but let's test it)
        # print(normalized_ranking)
        # Combine both probabilities to get our joint distribution
        # (Note: this can probably be more efficient than it currently is, but the input size is sufficiently small enough that it shouldn't matter)
        for x in joint_ranking:
            # Get the normalized probability for the item
            
            temp = [item for item in updated_ranking if item["name"] == x["name"]]
            # if len(temp) == 0:
            #     print(updated_ranking)
            #     print(joint_ranking)
            #     print(features)
                
            normalized_rank = temp[0]
            # print(normalized_rank)
            joint_distribution = phrase["confidence"] * normalized_rank["distribution"]
            # Slot the maximum value into the joint distribution
            if (joint_distribution > x["distribution"]):
                x["distribution"] = joint_distribution

    # Update the current distribution based on the new ranking
    # (Note: this can probably be more efficient than it currently is, but the input size is sufficiently small enough that it shouldn't matter)
    refined_ranking = []
    
    # print('Joint Ranking\n', sorted(joint_ranking, key=lambda i: i["distribution"], reverse=True))
    for value in ranking:
        updated_dist = [item for item in joint_ranking if item["name"] == value["name"]][0]
        refined_ranking.append({"name":value["name"],"distribution":((value["distribution"] * PREV_WEIGHT) + (updated_dist["distribution"] * UPDATED_WEIGHT))})

    # print(refined_ranking)
    # Return the sorted ranked list of probabilities
    return similarity.normalize(refined_ranking, normalization)



'''
----------get_avg_text_embedding----------
- This function gets the average text embedding for a given embedding
-----Inputs-----
- text - the text to parse
- embedding - the dictionary of embeddings
- configuration - the currently-active configuration
-----Output-----
- avg_embedding - the average embedding over all the recognized words
'''
def get_avg_text_embedding(text, embedding, configuration):
    embedding_name = configuration["embedding"]
    if "fasttext" in embedding_name:
        if "finetuned" in embedding_name:
            pass
        else:
            return fasttext.get_avg_text_vector(text, embedding["embedding"])

    elif "use" in embedding_name:
        if "finetuned" in embedding_name:
            pass
        else:
            return use.get_avg_text_vector(text, embedding["embedding"])
    elif "glove" in embedding_name:
        if "finetuned" in embedding_name:
            pass
        else:
            return glove.get_avg_text_vector(text, embedding["embedding"])
    elif "word2vec" in embedding_name:
        if "finetuned" in embedding_name:
            pass
        else:
            return word2vec.get_avg_text_vector(text, embedding["embedding"])
    elif "bert" in embedding_name:
        if "finetuned" in embedding_name:
            pass
        else:
            pass
    elif "roberta" in embedding_name:
        if "finetuned" in embedding_name:
            pass
        else:
            pass
    elif "xlnet" in embedding_name:
        if "finetuned" in embedding_name:
            pass
        else:
            pass
    print("No embedding detected")
    return