'''
- petel.py
- This file is meant to handle all of the PeTEL expression management for VIDS
'''


# External imports

# Internal imports
import src.core.interface.ranking as ranking
import src.core.interface.similarity as similarity

# Parameters
DEMOTION_WEIGHT = 1e-10 # This is the weight of a feature's current probability distribution after demotion (wrong guess)
BALANCE_WEIGHT = 100


class petel:
    '''
    ----------Constructor----------
    - Initializes all necessary values of the class
    -----Inputs-----
    - schema - the schema to use for the PeTEL expression
    - json_exp - the json expression to initialize the PeTEL expression to
    -----Output-----
    - self - the initialized PeTEL object, bound to a variable
    '''
    def __init__(self):
        self.flags = {"attribute":False, "filter" : False, "filter_operation": False, "aggregator": False, "prediction_window": False}
        self.valid = False
        self.active_feature = "attribute"
        self.counter = 0
        # "entity":[],
        self.rankings = {"attribute":[], "filter":[], "filter_operation":[], "aggregator":[], "prediction_window":[]}
            
    def __init__(self, schema, json_exp={}):
        if not json_exp:
            #"entity":False, 
            self.flags = {"attribute":{'active': True, 'completed': False, 'count': 0}, 
                            "filter" : {'active': False, 'completed': False, 'count': 0},
                            "filter_operation": {'active': False, 'completed': False, 'count': 0},
                            "aggregator": {'active': False, 'completed': False, 'count': 0},
                            "prediction_window": {'active': False, 'completed': False, 'count': 0}}
            self.valid = False
            self.active_feature = "attribute"
            self.counter = 0
            # "entity":[],
            self.rankings = {"attribute":[], "filter":[], "filter_operation":[], "aggregator":[], "prediction_window":[]}
            # print(self.flags)
            # Initialize the PeTEL object to the uniform distribution
            for feature in self.flags:
                # print(feature)
                for value in schema[feature]:
                    self.rankings[feature].append({"name":value["name"], "distribution":0})
            return
        self.flags = json_exp['flags']
        self.valid = json_exp['valid']
        self.active_feature = json_exp['active_feature']
        # print(json_exp['rankings'])
        self.rankings = json_exp['rankings']
        self.counter = json_exp['counter']
        return
    


    '''
    ----------update----------
    - Updates all ranked lists' distributions based on a new input
    -----Inputs-----
    - self - the PeTEL object to update (included intrinsically)
    - text - the user input to parse
    - schema - the currently-active user database schema
    - embedding - the currently-active embedding
    - configuration - the currently-active configuration
    -----Output-----
    - self - the bound PeTEL object's rankings' distributions are all updated based off the text and embedding
    '''
    def update(self, text, schema, embedding, configuration):
        i=0
        for key in self.flags:
            self.rankings[key] = ranking.update_rankings(text, schema[key], self.rankings[key], embedding, configuration)
        return

    
    '''
    ----------update_active----------
    - Updates the targeted ranked list's distribution based on a new input
    -----Inputs-----
    - self - the PeTEL object to update (included intrinsically)
    - text - the user input to parse
    - schema - the currently-active user database schema
    - embedding - the currently-active embedding
    - configuration - the currently-active configuration
    -----Output-----
    - self - the target ranking distribution of the bound PeTEL object is updated based off the text and embedding
    '''
    def update_active(self, text, schema, embedding, configuration):
        # This function will accept an additional confidence argument, and pass it along to the ranking function
        # print(self.rankings)
        self.rankings[self.active_feature] = ranking.update_joint_ranking(text, schema[self.active_feature], self.rankings[self.active_feature], embedding, configuration)
        # JOINT RANKING: To revert, comment out the above line, and uncomment the below line
        #self.rankings[self.active_feature] = ranking.update_rankings(text, schema[self.active_feature], self.rankings[self.active_feature], embedding, configuration)


    '''
    ----------demote_value----------
    - Lowers the target feature's probability, so it goes to the bottom of the list
    -----Inputs-----
    - self - the PeTEL object to update (included intrinsically)
    - index - the index of the value of the active feature to demote
    -----Output-----
    - self - the target value in the currently-active feature has a lowered probability in the distribution, pushing it to the bottom of the ranking
    '''
    def demote_value(self, index, normalization="softmax"):
        # for i in range(1, len(self.rankings[self.active_feature])):
        #     if i < len(self.rankings[self.active_feature])/2:
        #         self.rankings[self.active_feature][i]["distribution"] = self.rankings[self.active_feature][i]["distribution"] * (BALANCE_WEIGHT)
        
        self.rankings[self.active_feature][index]["distribution"] = self.rankings[self.active_feature][index]["distribution"] * DEMOTION_WEIGHT
        # print(self.rankings[self.active_feature][index]["distribution"])
        self.rankings[self.active_feature] = similarity.normalize(self.rankings[self.active_feature], normalization)
        
        # print(self.rankings[self.active_feature][index])

    '''
    ----------finished_feature----------
    - Description
    -----Inputs-----
    - input - description
    -----Output-----
    - output - description
    '''
    def finished_feature(self):# Can probably be streamlined
        # print(self.active_feature)
        for key in self.flags:
            if self.flags[key]['active'] and not self.flags[key]['completed']:
                self.flags[key]['completed'] = True
                self.flags[key]['active'] = False
            elif not self.flags[key]['active'] and not self.flags[key]['completed']:
                self.flags[key]['active'] = True
                self.active_feature = key
                return
        return


    '''
    ----------is_complete----------
    - Description
    -----Inputs-----
    - input - description
    -----Output-----
    - output - description
    '''
    def is_complete(self):
        for key in self.flags:
            if not self.flags[key]['completed']:
                return False
        return True