import torch
import nlp
from transformers import T5Tokenizer
import dataclasses
import logging
import os
import sys
from dataclasses import dataclass, field
from typing import Dict, List, Optional
import json
import numpy as np

from transformers import T5ForConditionalGeneration, T5Tokenizer, EvalPrediction
from transformers import (
    DataCollator,
    Trainer,
    TrainingArguments,
    set_seed,
)


tokenizer = T5Tokenizer.from_pretrained('t5-base')
logger = logging.getLogger(__name__)

# process the examples in input and target text format and the eos token at the end 
def add_eos_to_examples(example):
    example['input_text'] = 'concepts: %s </s>' % (example['concepts'])
    example['target_text'] = '%s </s>' % example['target']
    return example

# tokenize the examples
def convert_to_features(example_batch):
    input_encodings = tokenizer.batch_encode_plus(example_batch['input_text'], pad_to_max_length=True, max_length=512)
    target_encodings = tokenizer.batch_encode_plus(example_batch['target_text'], pad_to_max_length=True, max_length=16)

    encodings = {
        'input_ids': input_encodings['input_ids'], 
        'attention_mask': input_encodings['attention_mask'],
        'target_ids': target_encodings['input_ids'],
        'target_attention_mask': target_encodings['attention_mask']
    }

    return encodings

# load train and validation split of squad
train_dataset  = nlp.load_dataset('common_gen', split=nlp.Split.TRAIN)
valid_dataset = nlp.load_dataset('common_gen', split=nlp.Split.VALIDATION)

# map add_eos_to_examples function to the dataset example wise 
train_dataset = train_dataset.map(add_eos_to_examples)
# map convert_to_features batch wise
train_dataset = train_dataset.map(convert_to_features, batched=True)

valid_dataset = valid_dataset.map(add_eos_to_examples, load_from_cache_file=False)
valid_dataset = valid_dataset.map(convert_to_features, batched=True, load_from_cache_file=False)


# set the tensor type and the columns which the dataset should return
columns = ['input_ids', 'target_ids', 'attention_mask', 'target_attention_mask']
train_dataset.set_format(type='torch', columns=columns)
valid_dataset.set_format(type='torch', columns=columns)

print(len(train_dataset), len(valid_dataset))

torch.save(train_dataset, 'train_data.pt')
torch.save(valid_dataset, 'valid_data.pt')





# prepares lm_labels from target_ids, returns examples with keys as expected by the forward method
# this is necessacry because the trainer directly passes this dict as arguments to the model
# so make sure the keys match the parameter names of the forward method
@dataclass
class T2TDataCollator:
    def collate_batch(self, batch: List) -> Dict[str, torch.Tensor]:
        """
        Take a list of samples from a Dataset and collate them into a batch.
        Returns:
            A dictionary of tensors
        """
        input_ids = torch.stack([example['input_ids'] for example in batch])
        lm_labels = torch.stack([example['target_ids'] for example in batch])
        lm_labels[lm_labels[:, :] == 0] = -100
        attention_mask = torch.stack([example['attention_mask'] for example in batch])
        decoder_attention_mask = torch.stack([example['target_attention_mask'] for example in batch])
        

        return {
            'input_ids': input_ids, 
            'attention_mask': attention_mask,
            'lm_labels': lm_labels, 
            'decoder_attention_mask': decoder_attention_mask
        }


def main(args_dict):
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.

    # we will load the arguments from a json file, 
    #make sure you save the arguments in at ./args.json
    # model_args, data_args, training_args = parser.parse_json_file(json_file=os.path.abspath('args.json'))

    if (
        os.path.exists(args_dict.output_dir)
        and os.listdir(args_dict.output_dir)
        and args_dict.do_train
        and not args_dict.overwrite_output_dir
    ):
        raise ValueError(
            f"Output directory ({args_dict.output_dir}) already exists and is not empty. Use --overwrite_output_dir to overcome."
        )

    
    set_seed(args_dict.seed)

    # Load pretrained model and tokenizer
    #
    # Distributed training:
    # The .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    #args_dict.tokenizer_name if args_dict.tokenizer_name else
    tokenizer = T5Tokenizer.from_pretrained(
        args_dict.model_name_or_path,
        cache_dir=args_dict.cache_dir,
    )
    model = T5ForConditionalGeneration.from_pretrained(
        args_dict.model_name_or_path,
        cache_dir=args_dict.cache_dir,
    )

    # # Get datasets
    # print('loading data')
    # train_dataset  = torch.load(args_dict.train_file_path)
    # valid_dataset = torch.load(args_dict.valid_file_path)
    # print('loading done')
    
    training_args = TrainingArguments(
      output_dir=args_dict.output_dir,           # output directory
      num_train_epochs=2000,           # total number of training epochs
      per_device_train_batch_size=1,   # batch size per device during training, can increase if memory allows
      save_steps=500,                  # number of updates steps before checkpoint saves
      save_total_limit=5,              # limit the total amount of checkpoints and deletes the older checkpoints
      warmup_steps=500,                # number of warmup steps for learning rate scheduler
      weight_decay=0.01,               # strength of weight decay
      logging_dir='./logs',            # directory for storing logs
      logging_steps=10,
    )

    # Initialize our Trainer
    trainer = Trainer(
        model=model,
        args=training_args,
        train_dataset=train_dataset,
        eval_dataset=valid_dataset,
        data_collator=T2TDataCollator()
    )

    # Training
    if args_dict.do_train:
        trainer.train(
            model_path=args_dict.model_name_or_path if os.path.isdir(args_dict.model_name_or_path) else None
        )
        trainer.save_model()
        # For convenience, we also re-save the tokenizer to the same directory,
        # so that you can share your model easily on huggingface.co/models =)
        if trainer.is_world_master():
            tokenizer.save_pretrained(args_dict.output_dir)

    # Evaluation
    results = {}
    if args_dict.do_eval and args_dict.local_rank in [-1, 0]:
        logger.info("*** Evaluate ***")

        eval_output = trainer.evaluate()

        output_eval_file = os.path.join(args_dict.output_dir, "eval_results.txt")
        with open(output_eval_file, "w") as writer:
            logger.info("***** Eval results *****")
            for key in sorted(eval_output.keys()):
                logger.info("  %s = %s", key, str(eval_output[key]))
                writer.write("%s = %s\n" % (key, str(eval_output[key])))
    
        results.update(eval_output)
    
    return results
    
    
class args_dict:
    def __init__(self) -> None:
        self.num_cores= 8
        self.training_script = "train_t5_squad.py"
        self.model_name_or_path= "t5-base"
        self.max_len= 512 
        self.target_max_len= 16
        self.output_dir= "models"
        self.overwrite_output_dir= True
        self.per_gpu_train_batch_size= 8
        self.per_gpu_eval_batch_size= 8
        self.gradient_accumulation_steps= 4
        self.learning_rate= 1e-4
        self.tpu_num_cores= 8
        self.num_train_epochs= 4
        self.do_train= True
        self.seed= 123
        self.cache_dir = "models"



# with open('args.json', 'w') as f:
#   json.dump(args_dict, f)
arguments = args_dict()
main(arguments)

# import torch_xla.distributed.xla_multiprocessing as xmp

# xmp.spawn(_mp_fn, args=(), nprocs=8, start_method='fork')