"""
Copyright (c) Anonymous authors
All rights reserved.

MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""
from typing import List

import numpy as np

from cml.data.base import MultitaskDataset

from cml.data.dataset.base import DatasetGenerator


class Family(DatasetGenerator):
    def __init__(self, seed, fun_types: List = ["harm", "lin", "poly", "saw", "sine"]):
        super().__init__(seed)
        self.fun_types = fun_types

    def _rescale_linear(self, x, old_min, old_max, new_min, new_max):
        """
        Linearly rescale an arrary to the range [new_min, new_max].
        """
        slope = (new_max - new_min) / (old_max - old_min)
        intercept = new_min - slope * old_min

        return slope * x + intercept

    def sample_harmonics(self, num_tasks, num_samples):
        def harmonic(inputs, a1, a2, b1, b2, frequency):
            targets = a1 * np.sin(frequency * inputs + b1) + a2 * np.sin(2 * frequency * inputs + b2)

            return inputs, targets

        inputs = self.rng.uniform(low=-5.0, high=5.0, size=(num_tasks, num_samples, 1))
        frequency = self.rng.uniform(low=0.1, high=1.0, size=(num_tasks, 1, 1))
        a1 = self.rng.uniform(low=0.0, high=0.5, size=(num_tasks, 1, 1))
        a2 = self.rng.uniform(low=0.0, high=0.5, size=(num_tasks, 1, 1))
        b1 = self.rng.uniform(low=0.0, high=2.0 * np.pi, size=(num_tasks, 1, 1))
        b2 = self.rng.uniform(low=0.0, high=2.0 * np.pi, size=(num_tasks, 1, 1))

        inputs, targets = np.vectorize(harmonic)(inputs, a1, a2, b1, b2, frequency)

        return inputs, targets

    def sample_linears(self, num_tasks, num_samples):
        def linear(inputs, intercept, slope):
            return intercept + slope * inputs

        inputs = self.rng.uniform(low=-5.0, high=5.0, size=(num_tasks, num_samples, 1))
        intercepts = self.rng.uniform(low=-0.5, high=0.5, size=(num_tasks, 1, 1))
        slopes = self.rng.uniform(low=-0.1, high=0.1, size=(num_tasks, 1, 1))

        targets = np.vectorize(linear)(inputs, intercepts, slopes)

        return inputs, targets

    def sample_polynomials(self, num_tasks, num_samples):
        def polynomial(inputs, a, b, c):
            return a + b * inputs + c * inputs**2

        inputs = self.rng.uniform(low=-5.0, high=5.0, size=(num_tasks, num_samples, 1))
        a = self.rng.uniform(low=-5.0, high=5.0, size=(num_tasks, 1, 1))
        b = self.rng.uniform(low=-5.0, high=5.0, size=(num_tasks, 1, 1))
        c = self.rng.uniform(low=-5.0, high=5.0, size=(num_tasks, 1, 1))

        targets = np.vectorize(polynomial)(inputs, a, b, c)

        # Normalize targets to range -1, 1
        mins = np.expand_dims(np.min(targets, axis=1), -1)
        maxs = np.expand_dims(np.max(targets, axis=1), -1)
        targets = self._rescale_linear(targets, mins, maxs, -1.0, 1.0)

        return inputs, targets

    def sample_sawtooths(self, num_tasks, num_samples):
        def sawtooth(inputs, amplitude, phase):
            from scipy import signal
            return amplitude * signal.sawtooth(inputs - phase, width=0.5)

        inputs = self.rng.uniform(low=-5.0, high=5.0, size=(num_tasks, num_samples, 1))
        amplitude = self.rng.uniform(low=0.1, high=1.0, size=(num_tasks, 1, 1))
        phase = self.rng.uniform(low=0.0, high=2.0 * np.pi, size=(num_tasks, 1, 1))

        targets = np.vectorize(sawtooth)(inputs, amplitude, phase)

        return inputs, targets

    def sample_sinusoids(self, num_tasks, num_samples):
        def sinusoid(inputs, amplitude, phase):
            targets = amplitude * np.sin(inputs + phase)

            return targets

        inputs = self.rng.uniform(low=-5.0, high=5.0, size=(num_tasks, num_samples, 1))
        amplitudes = self.rng.uniform(low=0.1, high=1.0, size=(num_tasks, 1, 1))
        phases = self.rng.uniform(low=0.0, high=np.pi, size=(num_tasks, 1, 1))

        targets = np.vectorize(sinusoid)(inputs, amplitudes, phases)

        return inputs, targets

    def sample(self, num_tasks, num_samples):
        assert num_tasks % len(self.fun_types) == 0
        num_tasks_per_family = num_tasks // len(self.fun_types)

        # Sample tasks for each family in fun_types
        inputs, targets = [], []
        if "harm" in self.fun_types:
            x, y = self.sample_harmonics(
                num_tasks=num_tasks_per_family, num_samples=num_samples
            )
            inputs.append(x)
            targets.append(y)

        if "lin" in self.fun_types:
            x, y = self.sample_linears(
                num_tasks=num_tasks_per_family, num_samples=num_samples
            )
            inputs.append(x)
            targets.append(y)

        if "poly" in self.fun_types:
            x, y = self.sample_polynomials(
                num_tasks=num_tasks_per_family, num_samples=num_samples
            )
            inputs.append(x)
            targets.append(y)

        if "saw" in self.fun_types:
            x, y = self.sample_sawtooths(
                num_tasks=num_tasks_per_family, num_samples=num_samples
            )
            inputs.append(x)
            targets.append(y)

        if "sine" in self.fun_types:
            x, y = self.sample_sinusoids(
                num_tasks=num_tasks_per_family, num_samples=num_samples
            )
            inputs.append(x)
            targets.append(y)

        # Combine all tasks and randomly permute the order of tasks
        inputs = np.concatenate(inputs)
        targets = np.concatenate(targets)
        task_id = np.repeat(np.arange(len(self.fun_types)), num_tasks_per_family)
        idx = self.rng.permutation(num_tasks)

        return MultitaskDataset(x=inputs[idx], y=targets[idx], task_id=task_id[idx])


class Harmonic(Family):
    def __init__(self, seed):
        super().__init__(seed, fun_types=["harm"])


class Linear(Family):
    def __init__(self, seed):
        super().__init__(seed, fun_types=["lin"])


class Polynomial(Family):
    def __init__(self, seed):
        super().__init__(seed, fun_types=["poly"])


class Sawtooth(Family):
    def __init__(self, seed):
        super().__init__(seed, fun_types=["saw"])


class SinusoidFamily(Family):
    def __init__(self, seed):
        # NOTE: This sinusoid is slightly different from the canonical sinewave regression
        super().__init__(seed, fun_types=["sine"])
