"""
Copyright (c) Anonymous authors
All rights reserved.

MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""
import json
import os
import pickle

from jax._src.lib import pytree
from pathlib import Path
from typing import Union
from urllib.request import urlretrieve


def download_url(url, root, filename=None):
    root = root.expanduser()
    if filename is None:
        filename = os.path.basename(url)
    filepath = root / filename

    if filepath.is_file():
        return

    root.mkdir(exist_ok=True)
    urlretrieve(url, filename=filepath)
    return filepath


def get_asset_path(*args):
    jax_meta_folder = Path(__file__).parent.parent
    assets_folder = jax_meta_folder / 'data' / 'dataset' / 'assets'
    return assets_folder.joinpath(*args)


def get_asset(*args, dtype=None):
    filename = get_asset_path(*args)
    if not filename.exists():
        raise IOError(f'File not found: {filename}')

    if dtype is None:
        dtype = filename.suffix
        dtype = dtype[1:]

    if dtype == 'json':
        with open(filename, 'r') as f:
            data = json.load(f)
    else:
        raise NotImplementedError(f'Unknown type: {dtype}')
    return data


def load_pytree(path: Union[str, Path], suffix=".pickle") -> pytree:
    """
    Load pytree from pickle. See https://github.com/google/jax/issues/2116
    """
    path = Path(path)
    if not path.is_file():
        raise ValueError(f'Not a file: {path}')
    if path.suffix != suffix:
        raise ValueError(f'Not a {suffix} file: {path}')
    with open(path, 'rb') as file:
        data = pickle.load(file)
    return data


def save_pytree(path: Union[str, Path], data: pytree, overwrite: bool = False, suffix=".pickle"):
    """
    Save pytree using pickle. See https://github.com/google/jax/issues/2116
    """
    path = Path(path)
    if path.suffix != suffix:
        path = path.with_suffix(suffix)
    path.parent.mkdir(parents=True, exist_ok=True)
    if path.exists():
        if overwrite:
            path.unlink()
        else:
            raise RuntimeError(f'File {path} already exists.')
    with open(path, 'wb') as file:
        pickle.dump(data, file)


def save_dict_as_json(config, name, dir):
    """
    Store a dictionary as a json text file.
    """
    with open(os.path.join(dir, name + ".json"), 'w') as file:
        json.dump(config, file, sort_keys=True, indent=4, skipkeys=True,
                  default=str)
