"""
Copyright (c) Anonymous authors
All rights reserved.

MIT License

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the “Software”), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""
import abc
import math
from operator import add
from typing import Optional

import haiku as hk
import jax.numpy as jnp
import jax.tree_util as jtu
from cml.models.bank import Embedding, LinearBank, MLPBank, DeepBank

from cml.utils.pytree import PytreeReshaper, is_tuple_of_ints


class Hypernetwork(abc.ABC, hk.Module):
    """
    Abstract base class for all Hypernetworks.
    Args:
        params_target: dictionary with the params to be generated by the hypernetwork
        chunk_shape: the shape of a single chunk the weight_generator generates.
            The chunk shape needs to divide all target_shapes without remainder.
            For example if params_target have shapes=[(1, 50), (50, 50)],
            then valid chunk_shapes would be chunk_shape=(1, 50), chunk_shape=(1, 1)
            but chunk_shape != (3, 50) or chunk_shape != (50, 3) would be invalid.
            If set to `None`, the hypernetwork doesn't use chunking.
        embedding_dim: the embedding dimension
    """

    def __init__(
        self,
        params_target: dict,
        type: str,
        chunking: bool,
        num_templates: int,
        embedding_nonlinearity: str,
        embedding_dropout: Optional[float],
        embedding_norm_stop_grad: bool,
        embedding_normalization: bool,
        embedding_constant_init: bool,
        weight_generator_init: str,
        name: Optional[str] = None,
    ):
        super().__init__(name=name)

        self.type = type
        self.chunking = chunking
        self.num_templates = num_templates
        self.embedding_nonlinearity = embedding_nonlinearity
        self.embedding_dropout = embedding_dropout
        self.embedding_norm_stop_grad = embedding_norm_stop_grad
        self.embedding_normalization = embedding_normalization
        self.embedding_constant_init = embedding_constant_init
        self.weight_generator_init = weight_generator_init

        self.params_shape = jtu.tree_map(jnp.shape, params_target)
        self.num_chunks, self.dim_chunks = Hypernetwork.get_chunk_sizes(
            self.params_shape, self.chunking
        )
        self.reshaper = PytreeReshaper(self.params_shape)

    @property
    def weight_generator(self):
        if self.type == "linear":
            return LinearBank(
                self.num_templates,
                self.dim_chunks,
                self.weight_generator_init,
            )
        elif self.type == "mlp":
            return MLPBank(
                num_templates=self.num_templates,
                chunk_dim=self.dim_chunks,
                hidden_dims=[
                    64,
                ],
                init=self.weight_generator_init,
            )
        elif self.type == "deepmlp":
            return DeepBank(
                num_templates=self.num_templates,
                chunk_dim=self.dim_chunks,
                hidden_dim=32,
                init=self.weight_generator_init,
            )
        else:
            raise ValueError

    @property
    def embedding(self):
        return Embedding(
            num_embeddings=self.num_chunks,
            num_templates=self.num_templates,
            nonlinearity=self.embedding_nonlinearity,
            dropout_rate=self.embedding_dropout,
            normalization=self.embedding_normalization,
            normalization_stop_grad=self.embedding_norm_stop_grad,
            constant_init=self.embedding_constant_init,
        )

    @staticmethod
    def get_chunk_sizes(params_shape, chunking: bool):
        if chunking:
            # Chunking across layers of the target network
            num_chunks = len(params_shape)
            dim_chunks_all = []
            for params_shape_layer in params_shape.values():
                dim_chunks_tree = jtu.tree_map(
                    math.prod, params_shape_layer, is_leaf=is_tuple_of_ints
                )
                dim_chunks = jtu.tree_reduce(add, dim_chunks_tree)
                dim_chunks_all.append(dim_chunks)
            assert all([dim_chunks == d for d in dim_chunks_all])
        # elif chunking == "neuron":
        #     # Assuming chunk sizes divides all param shapes without remainder
        #     def num_chunks_fn(shape):
        #         return math.prod(jtu.tree_map(floordiv, shape, chunk_shape))

        #     num_chunks_tree = jtu.tree_map(num_chunks_fn, params_shape, is_leaf=is_tuple_of_ints)
        #     num_chunks = jtu.tree_reduce(add, num_chunks_tree)
        #     dim_chunks = math.prod(chunk_shape)
        else:
            # No chunking means all params are generated at once
            num_chunks = 1
            dim_chunks_tree = jtu.tree_map(math.prod, params_shape, is_leaf=is_tuple_of_ints)
            dim_chunks = jtu.tree_reduce(add, dim_chunks_tree)

        return num_chunks, dim_chunks

    def __call__(self, is_training: bool):
        params_flat = hk.vmap(self.weight_generator, split_rng=not hk.running_init())(
            self.embedding(is_training)
        )

        return self.reshaper(params_flat.reshape(-1))
