import math

import torch
import torch.nn as nn
import torch.nn.functional as F


# from https://github.com/kuangliu/pytorch-cifar/blob/master/models/densenet.py
class Bottleneck(nn.Module):
    def __init__(self, in_planes, growth_rate):
        super(Bottleneck, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(in_planes, 4*growth_rate, kernel_size=(1, 1), bias=False)
        self.bn2 = nn.BatchNorm2d(4*growth_rate)
        self.conv2 = nn.Conv2d(4*growth_rate, growth_rate, kernel_size=(3, 3), padding=(1, 1), bias=False)

    def forward(self, x):
        out = self.conv1(F.relu(self.bn1(x)))
        out = self.conv2(F.relu(self.bn2(out)))
        out = torch.cat([out, x], 1)
        return out


class Transition(nn.Module):
    def __init__(self, in_planes, out_planes):
        super(Transition, self).__init__()
        self.bn = nn.BatchNorm2d(in_planes)
        self.conv = nn.Conv2d(in_planes, out_planes, kernel_size=(1, 1), bias=False)

    def forward(self, x):
        out = self.conv(F.relu(self.bn(x)))
        out = F.avg_pool2d(out, 2)
        return out


class DenseNet(nn.Module):
    def __init__(self, name, block, n_blocks, num_classes, growth_rate=12, reduction=0.5):
        super(DenseNet, self).__init__()
        self.name = name
        self.num_classes = num_classes
        self.growth_rate = growth_rate

        num_planes = 2*growth_rate
        self.conv1 = nn.Conv2d(3, num_planes, kernel_size=(3, 3), padding=(1, 1), bias=False)

        self.dense1 = self._make_dense_layers(block, num_planes, n_blocks[0])
        num_planes += n_blocks[0] * growth_rate
        out_planes = int(math.floor(num_planes*reduction))
        self.trans1 = Transition(num_planes, out_planes)
        num_planes = out_planes

        self.dense2 = self._make_dense_layers(block, num_planes, n_blocks[1])
        num_planes += n_blocks[1] * growth_rate
        out_planes = int(math.floor(num_planes*reduction))
        self.trans2 = Transition(num_planes, out_planes)
        num_planes = out_planes

        self.dense3 = self._make_dense_layers(block, num_planes, n_blocks[2])
        num_planes += n_blocks[2] * growth_rate
        out_planes = int(math.floor(num_planes*reduction))
        self.trans3 = Transition(num_planes, out_planes)
        num_planes = out_planes

        self.dense4 = self._make_dense_layers(block, num_planes, n_blocks[3])
        num_planes += n_blocks[3] * growth_rate

        self.bn = nn.BatchNorm2d(num_planes)
        self.linear = nn.Linear(num_planes, num_classes)

    def _make_dense_layers(self, block, in_planes, n_block):
        layers = []
        for i in range(n_block):
            layers.append(block(in_planes, self.growth_rate))
            in_planes += self.growth_rate
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.conv1(x)
        out = self.trans1(self.dense1(out))
        out = self.trans2(self.dense2(out))
        out = self.trans3(self.dense3(out))
        out = self.dense4(out)
        out = F.avg_pool2d(F.relu(self.bn(out)), 4)
        out = out.view(out.size(0), -1)

        return self.linear(out)


def DenseNet121(num_classes):
    return DenseNet('densenet121', Bottleneck, [6, 12, 24, 16], num_classes=num_classes, growth_rate=32)


def DenseNet169(num_classes):
    return DenseNet('densenet169', Bottleneck, [6, 12, 32, 32], num_classes=num_classes, growth_rate=32)


def DenseNet201(num_classes):
    return DenseNet('densenet201', Bottleneck, [6, 12, 48, 32], num_classes=num_classes, growth_rate=32)


# For imagenet
def DenseNet161(num_classes):
    return DenseNet('densenet161', Bottleneck, [6, 12, 36, 24], num_classes=num_classes, growth_rate=48)


def densenet_cifar(num_classes):
    return DenseNet('densenet_cifar', Bottleneck, [6, 12, 24, 16], num_classes=num_classes, growth_rate=12)


def test():
    net = densenet_cifar(num_classes=1)
    x = torch.randn(1, 3, 32, 32)
    y = net(x)
    print(y)
